<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\User;
use Exception, Stripe;

class StripeService
{
  private $public_key, $secret_key, $statement_descriptor, $environment;
  private $stripe;
  
  public function paymentSetting()
  {
    $payment_gateway = Session()->get('payment_gateway');
    
    $this->public_key = $payment_gateway->public_key;
    $this->secret_key = $payment_gateway->secret_key;
    $this->statement_descriptor = $payment_gateway->statement_descriptor;
    $this->environment = $payment_gateway->environment;

    $this->stripe = new \Stripe\StripeClient($this->secret_key);
  }

  public function createCustomer($customerData, $item)
  {
    try {
      $this->paymentSetting();

      $customer = $this->stripe->customers->create([
        'description' => $item->item_name,
        'email' => $customerData->email,
        'phone' => $customerData->phone,
        'name' => $customerData->first_name. ' ' .$customerData->last_name,
        'address' => [
          "line1" => $customerData->address,
          "city" => $customerData->city,
          "country" => $customerData->country,
          "state" => $customerData->state,
        ],
        'metadata' => array(
          "token" => $item->token,
          "item" => $item->item_name,
          "first_name" => $customerData->first_name,
          "last_name" => $customerData->last_name,
          "email" => $customerData->email,
          "phone" => $customerData->phone,
          "company" => $customerData->company,
          "address" => $customerData->address,
          "city" => $customerData->city,
          "country" => $customerData->country,
          "zipcode" => $customerData->zipcode,
        )
      ]);
      
      return $customer;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function createPaymentIntent($item, $payment) {
    try {
      $this->paymentSetting();

      $paymentIntent = $this->stripe->paymentIntents->create([
        'description' => $item->item_name,
        'amount' => $item->price * 100,
        'currency' => $item->currencyCountry->code,
        'automatic_payment_methods' => [
          'enabled' => true,
          'allow_redirects' => 'never'
        ],
        'statement_descriptor' => $this->statement_descriptor,
        'setup_future_usage' => 'off_session',
      ]);
  
      // Payment Log here
  
      return $paymentIntent;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function attachCustomer($paymentMethod, $customerID) {
    try {
      $this->paymentSetting();
      $payment_method = $this->stripe->paymentMethods->retrieve($paymentMethod);
      // Attaching Customer with Payment Method
      $payment_method->attach([
        ['customer' => $customerID]
      ]);

      return $payment_method;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function getChargeObject($chargeId) {
    try {
      $this->paymentSetting();
      $charge = $this->stripe->charges->retrieve($chargeId);
      
      return $charge;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function getIntentObject($intentId) {
    try {
      $this->paymentSetting();
      $intent = $this->stripe->paymentIntents->retrieve($intentId);
      
      return $intent;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function createPaymentMethod($item) {
    try {
      $this->paymentSetting();

      $paymentMethod = $this->stripe->paymentMethods->create([
        'type' => 'card',
        'card' => [
          'number' => $item->cardNo,
          'exp_month' => $item->cardExp_month,
          'exp_year' => $item->cardExp_year,
          'cvc' => $item->card_cvc,
        ],
        'billing_details' => [
          'address' => [
            'city' => $item->city,
            'country' => $item->currency_country->aplha_code2,
            'line1' => $item->address,
            'postal_code' => $item->zipcode,
            'state' => $item->state
          ],
          'email' => $item->email,
          'name' => $item->first_name .' '. $item->last_name,
          'phone' => $item->phone,
        ]
      ]);
      
      return $paymentMethod;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function getPMObject($paymentMethodID) {
    try {
      $this->paymentSetting();
      $intent = $this->stripe->paymentMethods->retrieve($paymentMethodID);
      
      return $intent;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }

  public function updatePaymentIntent($paymentIntentID, $data) {
    try {
      $this->paymentSetting();
      $paymentIntent = $this->stripe->paymentIntents->update($paymentIntentID, $data);
      
      return $paymentIntent;
    } catch(Exception $e) {
      StripeErrorHandler::handle($e);
    }
  }
}
