<?php

namespace App\Services\Payment;

use App\Exceptions\StripeException;
use App\Helper\GeneralHelper;
use App\Helper\Helper\NotificationHelper;
use App\Helper\Helper\StripeHelper;
use App\Repositories\Payment\PaymentInterface;
use App\Services\CustomerService;
use App\Services\LogService;
use App\Services\Payment\Stripe\Customer;
use App\Services\Payment\Stripe\PaymentIntent;
use App\Services\Payment\Stripe\PaymentMethod;
use App\Services\Payment\Stripe\Charge;
use App\Services\StripeErrorHandler;
use Carbon\Carbon;
use Exception;

class PaymentService
{
    private $paymentRepository, $logRepository, $customerService;

    // Stripe Services
    private $stripeCustomer, $stripeCharge, $stripePaymentIntent, $stripePaymentMethod;

    function __construct(
        PaymentInterface $payment,
        LogService $log,
        Customer $stripeCustomer,
        PaymentIntent $stripePaymentIntent,
        PaymentMethod $stripePaymentMethod,
        Charge $stripeCharge,
        CustomerService $customerService
    ) {
        $this->paymentRepository = $payment;
        $this->logRepository = $log;
        $this->customerService = $customerService;

        // Stripe Services
        $this->stripeCustomer = $stripeCustomer;
        $this->stripeCharge = $stripeCharge;
        $this->stripePaymentIntent = $stripePaymentIntent;
        $this->stripePaymentMethod = $stripePaymentMethod;
    }

    public function initiate($customerID, $link, $ip, $gatewayMessage, $status = "")
    {
        $data = [];
        try {
            $data = [
                "customer_id" => $customerID,
                "payment_link_id" => $link->id,
                "price" => $link->price,
                "discount" => $link->discount,
                "currency" => $link->currency,
                "ip" => $ip,
                "converted_amount" => $link->price,
                "last_four" => 0000,
                "comment" => $link->comment,
                "status" => $status != "" ? $status : 3,
                "created_at" => Carbon::now()
            ];

            if (str_contains($link->gateway->gateway, 'three_step')) {
                $data['price'] = ($link->price - 1);
            }

            $payment = $this->paymentRepository->create($data);

            $this->logRepository->log('payment', [
                'activity' => 'Payment is started',
                'type' => 'payment.started',
                'request' => response()->json($data)->content(),
                'response' => response()->json([])->content(),
                'code' => 200,
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            ]);

            return $payment;
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {
            StripeErrorHandler::handle($e);
        }
    }

    public function addCustomer($customerData, $item_detail)
    {
        $stripeCustomer = [];
        try {
            $stripeCustomer = (object) $this->stripeCustomer->create($customerData, $item_detail);

            $customerData->stripe_customer_id = $stripeCustomer->id;
            $customerData->stripe = response()->json($stripeCustomer)->content();

            $customer = $this->customerService->create($customerData, $item_detail->id);

            if (!$customer['existing_customer']) {

                $log = array(
                    'activity' => 'Customer was Created',
                    'type' => 'customer.created',
                    'request' => response()->json($customerData)->content(),
                    'response' => response()->json($customer)->content(),
                    'code' => response()->json($customer)->status(),
                    'loggable_id' => $customer['id'],
                    'created_by' => $item_detail->created_by,
                );
                $this->logRepository->log('customer', $log);
            }

            return [
                'customer' => (object) $stripeCustomer,
                'internal' => $customer['id']
            ];
        } catch (StripeException $e) {
            StripeErrorHandler::handle($e);
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function addPaymentMethod($paymentID, $paymentDetails, $link, $customer)
    {
        try {
            $payment_method = $this->stripePaymentMethod->create($paymentDetails);

            $this->stripeCustomer->attach($payment_method->id, $customer['stripe_id']);

            $payment_method = $this->stripePaymentMethod->get($payment_method->id);

            $authorizedLog = array(
                'activity' => 'Payment is Authorized',
                'type' => 'payment.attach_payment_method',
                'request' => response()->json([
                    'payment' => [
                        'id' => $paymentID,
                    ],
                    'payment_method' => $payment_method,
                    'customer' => [
                        'id' => $customer['stripe_id']
                    ],
                    'created_by' => $link->created_by
                ])->content(),
                'response' => response()->json($payment_method)->content(),
                'code' => response()->json($payment_method)->status(),
                'loggable_id' => $paymentID,
                'created_by' => $link->created_by,
            );

            $this->logRepository->log('payment', $authorizedLog);

            $customerLog = array(
                'activity' => 'A new ' . ucwords($payment_method->card?->brand) . ' card ending in ' . ucwords($payment_method->card?->last4) . ' was added',
                'type' => 'customer.payment_method',
                'request' => response()->json($payment_method)->content(),
                'response' => response()->json($payment_method)->content(),
                'code' => response()->json($payment_method)->status(),
                'loggable_id' => $customer['internal'],
                'created_by' => $link->created_by,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->customerService->update([
                'stripe_pm_id' => $payment_method->id,
                'stripe_payment_method' => response()->json($payment_method)->content()
            ], $customer['internal']);

            return $payment_method;
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {
            StripeErrorHandler::handle($e);
        }
    }

    public function process($payment, $customer, $link, $paymentDetails, $statement_descriptor, $gatewayMessage = "")
    {
        try {
            $metadata = [
                "token" => $paymentDetails->token,
                "first_name" => $paymentDetails->first_name,
                "last_name" => $paymentDetails->last_name,
                "email" => $paymentDetails->email,
                "phone" => $paymentDetails->phone,
                "company" => $paymentDetails->company,
                "address" => $paymentDetails->address,
                "city" => $paymentDetails->city,
                "state" => $paymentDetails->state,
                "country" => $link->currencyCountry->aplha_code3,
                "brand_descriptor" => $paymentDetails->brand_descriptor
            ];

            // Creating Payment Intent
            $paymentIntent = $this->stripePaymentIntent->create($link, $customer['stripe_id'], $metadata);

            $intentLog = array(
                'activity' => 'Payment is Initiated',
                'type' => 'payment.initiated',
                'request' => response()->json($link)->content(),
                'response' => response()->json($paymentIntent)->content(),
                'code' => response()->json($paymentIntent)->status(),
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            );

            //$this->logRepository->log('payment', $intentLog);

            $this->logRepository->eventStatus($paymentIntent->status, $payment->id, response()->json($paymentIntent)->content(), $link->created_by);

            $intentLog['type'] = 'payment.intent.created';
            $this->logRepository->log('payment', $intentLog, "", 1);

            // Creating Payment Method
            $payment_method = $this->addPaymentMethod($payment->id, $paymentDetails, $link, $customer);

            $paymentIntent = $this->stripePaymentIntent->update($paymentIntent->id, [
                'customer' => $customer['stripe_id'],
                'payment_method' => $payment_method->id,
                'statement_descriptor' => $statement_descriptor,
                'metadata' => [
                    "token" => $paymentDetails->token,
                    "first_name" => $paymentDetails->first_name,
                    "last_name" => $paymentDetails->last_name,
                    "email" => $paymentDetails->email,
                    "phone" => $paymentDetails->phone,
                    "company" => $paymentDetails->company,
                    "address" => $paymentDetails->address,
                    "city" => $paymentDetails->city,
                    "state" => $paymentDetails->state,
                    "country" => $link->currencyCountry->aplha_code3,
                    "brand_descriptor" => $paymentDetails->brand_descriptor
                ],
            ]);

            $intentLog = array(
                'activity' => 'Payment Intent Updated',
                'type' => 'payment.intent.updated',
                'request' => response()->json([])->content(),
                'response' => response()->json($paymentIntent)->content(),
                'code' => response()->json($paymentIntent)->status(),
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            );
            $this->logRepository->log('payment', $intentLog);

            $this->logRepository->eventStatus($paymentIntent->status, $payment->id, response()->json($paymentIntent)->content(), $link->created_by);

            $paymentIntent = $paymentIntent->confirm();

            return [
                'payment_intent' => $paymentIntent,
                'payment_method' => $payment_method
            ];
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {
            StripeErrorHandler::handle($e);
        }
    }

    public function afterpay_process($payment, $customer, $link, $paymentDetails, $statement_descriptor, $gatewayMessage = "")
    {
        try {
            $metadata = [
                "token" => $paymentDetails->token,
                "first_name" => $paymentDetails->first_name,
                "last_name" => $paymentDetails->last_name,
                "email" => $paymentDetails->email,
                "phone" => $paymentDetails->phone,
                "company" => $paymentDetails->company,
                "address" => $paymentDetails->address,
                "city" => $paymentDetails->city,
                "state" => $paymentDetails->state,
                "country" => $link->currencyCountry->aplha_code3,
                "brand_descriptor" => $paymentDetails->brand_descriptor
            ];

            // Creating Payment Intent
            $paymentIntent = $this->stripePaymentIntent->create($link, $customer['stripe_id'], $metadata);

            $intentLog = array(
                'activity' => 'Payment is Initiated',
                'type' => 'payment.initiated',
                'request' => response()->json($link)->content(),
                'response' => response()->json($paymentIntent)->content(),
                'code' => response()->json($paymentIntent)->status(),
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            );

            $this->logRepository->log('payment', $intentLog);
            $this->logRepository->eventStatus($paymentIntent->status, $payment->id, $intentLog['response'], $link->created_by);

            return [
                'payment_intent' => $paymentIntent,
            ];
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {
            StripeErrorHandler::handle($e);
        }
    }

    public function generateFile($token, $data, $paymentID, $createdBy)
    {
        try {
            if (!empty($data->cardNo)) {
                $data->cardNo = StripeHelper::maskNumber($data->cardNo);

                StripeHelper::removeKey($data, 'card_number');
                StripeHelper::removeKey($data, 'card_cvc');
                StripeHelper::removeKey($data, 'card_date');
            }

            $intentLog = array(
                'activity' => 'A request to encrypt data is initiated',
                'type' => 'payment.file.initiated',
                'request' => response()->json($data)->content(),
                'response' => response()->json([]),
                'code' => 200,
                'loggable_id' => $paymentID,
                'created_by' => $createdBy,
            );

            $this->logRepository->log('payment', $intentLog, "", 1);

            $file = StripeHelper::createFile($token, $data);

            if ($file) {
                $encrpted = StripeHelper::readFile($token);

                $intentLog = array(
                    'activity' => 'Data has been posted',
                    'type' => 'payment.file.posted',
                    'request' => response()->json($data)->content(),
                    'response' => response()->json($encrpted)->content(),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentLog);

                $intentLog = array(
                    'activity' => 'A new file creation request is completed',
                    'type' => 'payment.file.completed',
                    'request' => response()->json($data)->content(),
                    'response' => response()->json([]),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentLog, "", 1);
            } else {
                throw new Exception('File not created');
            }
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {

            $intentLog = array(
                'activity' => 'Failed to store encrypted file',
                'type' => 'payment.file.failed',
                'request' => response()->json($data),
                'response' => response()->json([
                    'code' => $e->getCode(),
                    'message' => $e->getMessage()
                ]),
                'code' => 500,
                'loggable_id' => $paymentID,
                'created_by' => $createdBy,
            );

            $this->logRepository->log('payment', $intentLog);

            return false;
            // throw $e;
        }
    }

    public function stripeChargeDetails($paymentID, $paymentIntentID, $customerID, $createdBy)
    {
        try {
            $data['intent'] = $this->stripePaymentIntent->get($paymentIntentID);
            $this->logRepository->eventStatus($data['intent']->status, $paymentID, response()->json($data['intent'])->content(), $createdBy);

            if (!empty($data['intent']) && !empty($data['intent']->latest_charge)) {
                $charge = $this->stripeCharge->get($data['intent']->latest_charge);
                $data['charge'] = response()->json($charge)->content();

                $intentLog = array(
                    'activity' => 'Payment is Charged',
                    'type' => 'payment.intent.charge',
                    'request' => response()->json([])->content(),
                    'response' => response()->json($charge)->content(),
                    'code' => response()->json($charge)->status(),
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentLog, "", 1);

                $intentSucceededLog = array(
                    'activity' => 'Payment Intent',
                    'type' => 'payment.intent.succeeded',
                    'request' => response()->json([])->content(),
                    'response' => response()->json($data['intent'])->content(),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentSucceededLog, "", 1);

            }

            if (!empty($data['intent']) && !empty($data['intent']->payment_method)) {
                $paymentMethod = $this->stripePaymentMethod->get($data['intent']->payment_method);

                $data['paymentMethod'] = response()->json($paymentMethod)->content();

                $this->customerService->update([
                    'stripe_pm_id' => $paymentMethod->id,
                    'stripe_payment_method' => $data['paymentMethod']
                ], $customerID);
            }

            $data['intent'] = response()->json($data['intent'])->content();
            $this->paymentRepository->update($paymentID, $data);


            return $data;
        } catch (StripeException $e) {
            return false;
        }
    }

    public function threestep_point_transaction($payment, $link, $customer)
    {
        $newPayment = (object) ["id" => 0];

        try {
            $data = [
                "customer_id" => $payment->customer_id,
                "payment_link_id" => $link->id,
                "price" => 0.5,
                "currency" => $link->currency,
                "ip" => $payment->ip,
                "paymentMethod" => $payment->paymentMethod,
                "last_four" => $payment->last_four,
                "device" => $payment->device,
                "location" => $payment->location,
                "bin" => $payment->bin,
                "converted_amount" => 0.5,
                "comment" => $payment->comment,
                "status" => 3,
                "created_at" => Carbon::now()
            ];

            $newPayment = $this->paymentRepository->create($data);

            $this->logRepository->log('payment', [
                'activity' => 'Payment is started',
                'type' => 'payment.started',
                'request' => response()->json($data)->content(),
                'response' => response()->json([])->content(),
                'code' => 200,
                'loggable_id' => $newPayment->id,
                'created_by' => $link->created_by,
            ]);

            $metadata = [
                "token" => $link->token,
                "first_name" => $customer->first_name,
                "last_name" => $customer->last_name,
                "email" => $customer->email,
                "phone" => $customer->phone,
                "company" => $customer->company,
                "address" => $customer->address,
                "city" => $customer->city,
                "state" => $customer->state,
                "country" => $link->currencyCountry->aplha_code3,
                "brand_descriptor" => $link->gateway->statement_descriptor
            ];

            $intent = GeneralHelper::decodeJSON($payment->intent);

            if ($intent) {
                // Creating Payment Intent
                $paymentIntent = $this->stripePaymentIntent->create($link, $intent->customer, $metadata, 0.5, $intent->payment_method);

                $this->logRepository->eventStatus($paymentIntent->status, $newPayment->id, response()->json($paymentIntent)->content(), $link->created_by);
                $intentLog = array(
                    'activity' => 'Payment is Initiated',
                    'type' => 'payment.initiated',
                    'request' => response()->json($link)->content(),
                    'response' => response()->json($paymentIntent)->content(),
                    'code' => response()->json($paymentIntent)->status(),
                    'loggable_id' => $newPayment->id,
                    'created_by' => $link->created_by,
                );

                $this->logRepository->log('payment', $intentLog);

                $paymentIntent = $this->stripePaymentIntent->update($paymentIntent->id, [
                    'statement_descriptor' => $link->gateway->statement_descriptor,
                    'metadata' => $metadata,
                ]);
                $paymentIntent = $paymentIntent->confirm();

                $this->logRepository->eventStatus($paymentIntent->status, $newPayment->id, response()->json($paymentIntent)->content(), $link->created_by);

                if ($paymentIntent->status == 'succeeded') {
                    $charge = $this->stripeCharge->get($paymentIntent->latest_charge);

                    $intentLog = array(
                        'activity' => 'Payment is Charged',
                        'type' => 'payment.intent.charge',
                        'request' => response()->json([])->content(),
                        'response' => response()->json($charge)->content(),
                        'code' => response()->json($charge)->status(),
                        'loggable_id' => $newPayment->id,
                        'created_by' => $link->created_by,
                    );

                    $this->logRepository->log('payment', $intentLog, "", 1);

                    $intentSucceededLog = array(
                        'activity' => 'Payment Intent',
                        'type' => 'payment.intent.succeeded',
                        'request' => response()->json([])->content(),
                        'response' => response()->json($paymentIntent)->content(),
                        'code' => 200,
                        'loggable_id' => $newPayment->id,
                        'created_by' => $link->created_by,
                    );

                    $this->logRepository->log('payment', $intentSucceededLog, "", 1);

                    $paymentMethod = $this->stripePaymentMethod->get($paymentIntent->payment_method);
                    $newPayment->paymentMethod = response()->json($paymentMethod)->content();

                    $newPayment->charge_id = $charge->id;
                    $newPayment->charge = response()->json($charge)->content();

                    $newPayment->intent_id = $paymentIntent->id;
                    $newPayment->intent = response()->json($paymentIntent)->content();

                    $this->logRepository->log('payment', [
                        'activity' => 'Payment succeeded',
                        'type' => 'success',
                        'request' => response()->json($link)->content(),
                        'response' => response()->json($charge)->content(),
                        'code' => 200,
                        'loggable_id' => $newPayment->id,
                        'created_by' => $link->created_by,
                    ]);

                    $newPayment->status = 1;
                    $newPayment->save();

                    NotificationHelper::notify('payment', 'Payment-Received: Amount of ' . $link->currencyCountry->symbol . $payment->price . ' ' . 'has been received', [
                        'id' => $newPayment->id,
                        'price' => 0.5,
                        'message' => 'Payment successful',
                    ], $link->created_by);
                }
            }
        } catch (StripeException $e) {
            $this->logRepository->log('payment', [
                'activity' => 'Payment Failed because of ' . (!empty($e->getCode()) ? $e->getCode() . ' | ' . $e->getMessage() : $e->getMessage()),
                'type' => 'failed',
                'request' => response()->json($link)->content(),
                'response' => response()->json($e->getError())->content(),
                'code' => 402,
                'loggable_id' => ($newPayment->id > 0) ? $newPayment->id : $payment->id,
                'created_by' => $link->created_by,
            ]);

            NotificationHelper::notify('payment-failed', 'Payment-Failed: Amount of ' . $link->currencyCountry->symbol . '0.5 ' . 'has been failed', [
                'id' => $payment->id,
                'price' => $payment->price,
                'request' => response()->json([
                    'payment' => response()->json($payment),
                    'link' => response()->json($link),
                    'customer' => response()->json($customer),
                ])->content(),
                'response' => response()->json($e->getError())->content(),
                'code' => 402
            ], $link->created_by);

            if ($newPayment->status == 3) {
                $newPayment->status = 2;
                $newPayment->save();
            }
        }
    }
}
