<?php

namespace App\Repositories\Role;

use App\Models\Role;
use Illuminate\Support\Arr;
use Exception;
use Spatie\Permission\Models\Role as ModelsRole;
use Spatie\Permission\Models\Permission;

class RoleRepository implements RoleInterface
{

    // Add or Update role
    public function addOrUpdate(...$args)
    {
        try {
            $id = $args[0]['id'] ?? null;
            $role = Role::findOrNew($id);
            $isNewRole = $role->id == null;

            if ($isNewRole) {
                $role->name = $args[0]['name'];
                $role->description = $args[0]['description'];
                $role->color = $args[0]['color'];
                $role->created_by = $args[0]['created_by'];
                $role->guard_name = 'web';
                $role->admin_rights = $args[0]['admin_rights'];
            } else {
               
                $role->name = $args[0]['name'];
                $role->description = $args[0]['description'];
                $role->color = $args[0]['color'];
                $role->updated_by = $args[0]['updated_by'];
                $role->guard_name = 'web';
                $role->admin_rights = $args[0]['admin_rights'] ? 1 : 0;
            }

            $role->save();

            $assignToRole = ModelsRole::find($role->id);

            if (!empty($args[0]['admin_rights']) && $args[0]['admin_rights']) {
                // Retrieve all permissions associated with the admin role
                $allPermissions = Permission::all();
                $excludedPermissions = [
                    'BlockList-Create', 'BlockList-Delete', 'Refund-View', 'Refund-Create', 
                    'Role-Create', 'Role-Edit', 'Role-View', 'Role-Delete'
                ];

                // Filter out the excluded permissions
                $permissionsToAssign = $allPermissions->filter(function($permission) use ($excludedPermissions) {
                    return !in_array($permission->name, $excludedPermissions);
                });

                // Assign admin permissions to the new role
                $assignToRole->syncPermissions($permissionsToAssign);
                
            } else if (isset($args[0]['admin_rights']) && !$args[0]['admin_rights']) {
                // Remove all permissions if admin_rights is false
                $assignToRole->syncPermissions([]);

                 $includedPermissions = [
                    'BlockList-View', 'BlockList-Create', 'BlockList-Edit', 
                    'Coupon-View', 'Coupon-Create', 'Coupon-Edit', 
                    'Customer-View', 'Customer-Create', 'Gateway-View', 'Gateway-Create', 
                    'ImportExport-View', 'ImportExport-Create', 
                    'Invoice-View', 'Invoice-Create', 'Invoice-Delete', 
                    'Notes-View', 'Notes-Create', 'Notes-Edit', 
                    'Payment-View', 
                    'PaymentLinkGenerator-View', 'PaymentLinkGenerator-Create', 
                    'Product-View', 
                    'TwoStep-Create', 'TwoStep-Edit'
                ];

                $permissions = Permission::whereIn('name', $includedPermissions)->get();
                $assignToRole->syncPermissions($permissions);
            }
    

            $data['success'] = true;
            $data['id'] = $role->id;

            return $data;
        } catch (Exception $ex) {
            $data['success'] = false;
            $data['message'] = $ex->getMessage();

            return $data;
        }
    }

    // Detail of role
    public function role($id)
    {
        return Role::find($id);
    }

    // list of roles
    public function list($select = "")
    {
        return Role::all();
    }

    // Archived role
    public function softDeleteRole($id)
    {
        try {
            $role = Role::find($id);
            if (!empty($role)) {
                $role->status = 2;
                $role->save();
                $role->delete(); // Soft delete
                $data['id'] = $role->id;
                $data['success'] = true;
                $data['message'] = 'Role Archived Successfully';

                return $data;
            }
        } catch (Exception $ex) {
            $data['success'] = false;
            $data['message'] = $ex->getMessage();

            return $data;
        }
    }

    // Permanent Delete role
    public function forceDeleteRole($id)
    {
        try {
            $role = Role::withTrashed()->find($id);
            if (!empty($role)) {
                $role->forceDelete(); // Permanent delete
                $data['success'] = true;
                $data['message'] = 'Role Deleted Successfully';

                return $data;
            }
        } catch (Exception $ex) {
            $data['success'] = false;
            $data['message'] = $ex->getMessage();

            return $data;
        }
    }

    // Restore role
    public function restoreRole($id)
    {
        try {
            $role = Role::withTrashed()->find($id);
            if (!empty($role)) {
                $role->restore(); // Restore the soft-deleted record
                $role->status = 1;
                $role->save();
                $data['id'] = $role->id;
                $data['success'] = true;
                $data['message'] = 'Role Restored Successfully';

                return $data;
            }
        } catch (Exception $ex) {
            $data['success'] = false;
            $data['message'] = $ex->getMessage();

            return $data;
        }
    }

    // Add role metadata
    public function updateRoleMetadata(...$args)
    {
        try {
            $id = $args[0]['id'] ?? null;
            $role = Role::find($id);

            if ($args[0]['metadata']) {
                $metadata = $args[0]['metadata'];

                $metadata = array_filter($metadata, function ($item) {
                    return $item['key'] !== null && $item['value'] !== null;
                });

                $metadata = Arr::pluck($metadata, 'value', 'key');
                $role->metadata = json_encode($metadata);
                $role->updated_by = $args[0]['updated_by'];
            }

            $role->save();

            $data['success'] = true;
            $data['id'] = $role->id;

            return $data;
        } catch (Exception $ex) {
            $data['success'] = false;
            $data['message'] = $ex->getMessage();

            return $data;
        }
    }
}
