<?php

namespace App\Repositories\Export;

use App\Exports\DynamicExport;
use Illuminate\Database\Eloquent\Collection;
use Maatwebsite\Excel\Facades\Excel;

class ExportRepository implements ExportInterface
{
    private $headRow, $data, $processed, $allColumns;

    public function headings($headings)
    {
        $this->headRow = $headings;
    }

    public function process($modelData, $payload)
    {
        $selectedColumns = $payload['columns'];
        $this->allColumns = $this->getAllColumns($this->headRow);

        // Process the data and select the columns
        $processedData = $modelData->map(function ($item) use ($selectedColumns) {
            $rowData = [];
            foreach ($selectedColumns as $column) {
                $value = $this->getColumnValue($item, $column);
                $rowData[$column] = $value;
            }
            return $rowData;
        });

        $rows = [];
        // Prepare data with custom headings
        $dataWithHeadings = [];
        foreach ($selectedColumns as $column => $heading) {
            $dataWithHeadings[] = $this->getHeaderText($heading); // Add heading to the first row
        }
        $rows[] = $dataWithHeadings;
        foreach ($processedData as $row) {
            $rowData = [];
            foreach ($selectedColumns as $column => $heading) {
                $rowData[] = $row[$heading] ?? null;
            }
            $rows[] = $rowData; // Add data row
        }

        $this->processed = true;

        $this->data = $rows;
    }

    public function export()
    {
        if (!$this->processed) {
            throw new \RuntimeException('Cannot export data before processing');
        }

        $collection = new Collection($this->data);
        $export = new DynamicExport($collection->toArray());
        $fileName =  uniqid() . '.xlsx';
        // $filePath = storage_path('app/exports/' . $fileName);
        Excel::store($export, 'public/exports/'.$fileName);

        $path = app()->environment('production')
            ? asset('storage/app/public/exports/' . $fileName) // For production environment
            : asset('storage/exports/' . $fileName); // For local environment
        
        // Return the path to the stored file
        return [
            'fileName' => $fileName,
            'path' => $path
        ];
    }

    private function getColumnValue($item, $column)
    {
        $keys = explode('.', $column);
        $value = $item;

        foreach ($keys as $key) {
            // Handle JSON string stored as long_text
            if (is_string($value) && $this->is_json($value)) {
                $value = json_decode($value);
            }

            if (is_array($value) || $value instanceof \ArrayAccess) {
                $value = $value[$key] ?? null;
            } elseif (is_object($value)) {
                // Handle case where value is an object
                if (isset($value->{$key})) {
                    if (is_string($value->{$key}) && $this->is_json($value->{$key})) {
                        // Decode JSON string
                        $value = json_decode($value->{$key});
                    } else {
                        $value = $value->{$key};
                    }
                } else {
                    return null; // Key does not exist, return null
                }
            } else {
                return null; // Value is neither an array nor an object, return null
            }
        }

        return $value;
    }


    // Helper function to check if a string is a valid JSON
    private function is_json($string)
    {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }

    private function getAllColumns($exportColumns)
    {
        $columns = [];

        foreach ($exportColumns['columns'] as $column) {
            $columns[] = $column;
        }

        if (!empty($exportColumns['related'])) {
            foreach ($exportColumns['related'] as $related) {
                $columns = array_merge($columns, $this->getAllColumns($related));
            }
        }

        return $columns;
    }

    private function getHeaderText($key)
    {
        $result = null;

        foreach ($this->allColumns as $item) {
            if ($item['key'] === $key) {
                $result = $item['text'];
                break;
            }
        }

        if ($result !== null) {
            return $result;
        } else {
            return '';
        }
    }
}
