<?php

namespace App\Http\Requests;

use Carbon\Carbon;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class PaypalPaymentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $currentYear = Carbon::now()->format('y');

        return [
            'firstname' => 'required',
            'lastname' => 'required',
            'clientemail' => ['required', 'email'],
            'phonenum' => 'required',
            'address' => 'required',
            'companyname' => 'required',
            'country' => 'required',
            'statename' => 'required',
            'city' => 'required',
            'zipcode' => 'required',
            'card_type' => ['required', Rule::in(['creditCard'])],
            'cardNo' => [
                Rule::requiredIf(function () {
                    return $this->input('card_type') === 'creditCard';
                }),
                'digits_between:14,16'
            ],
            'month' => [
                Rule::requiredIf(function () {
                    return $this->input('card_type') === 'creditCard';
                }),
                'digits:2'
            ],
            'cardValidity' => [
                Rule::requiredIf(function () {
                    return $this->input('card_type') === 'creditCard';
                }),
                function ($attribute, $value, $fail) {
                    if (!$value) {
                        $fail("The card number is invalid. Check the card details or use a different card.");
                    }
                }
            ],
            'year' => [
                Rule::requiredIf(function () {
                    return $this->input('card_type') === 'creditCard';
                }),
                'digits:4',
                function ($attribute, $value, $fail) use ($currentYear) {
                    if ($currentYear > $value) {
                        $fail("The card's expiration year is incorrect.");
                    }
                },
            ],
            'cvc' => [
                Rule::requiredIf(function () {
                    return $this->input('card_type') === 'creditCard';
                }),
                //'digits_between:2,4',
            ],
            'coupon_id' => [
                'nullable',
                Rule::requiredIf(function () {
                    return $this->input('coupon_id') > 0;
                }),
                'integer',
                'min:1',
                Rule::exists('coupons', 'id')
            ],
        ];
    }

    public function messages()
    {
        return [
            'cardNo.required' => 'The card number is invalid. ',
            'cardNo.digits_between' => 'The card number is invalid. ',
            'cardValidity.required' => "The card number is invalid. Check the card details or use a different card.",
            'month.required' => "The card's expiration month is incorrect.",
            'month.digits' => "The card's expiration month is incorrect.",
            'year.digits' => "The card's expiration year is incorrect.",
            'year.required' => "The card's expiration year is incorrect.",
            'cvc.required' => "Check the card's security code or use a different card.",
            'cvc.digits_between' => "Check the card's security code or use a different card.",
        ];
    }

    function validateCardNumber($cardNumber)
    {
        // Remove any non-digit characters from the card number
        $cleanedCardNumber = preg_replace('/\D/', '', $cardNumber);

        // Convert the card number to an array of digits
        $digits = str_split($cleanedCardNumber);

        // Double every second digit starting from the right
        for ($i = count($digits) - 2; $i >= 0; $i -= 2) {
            $digits[$i] *= 2;
            if ($digits[$i] > 9) {
                $digits[$i] -= 9;
            }
        }

        // Sum all the digits
        $sum = array_sum($digits);

        // Check if the sum is divisible by 10
        return $sum % 10 === 0;
    }
}
