<?php

namespace App\Http\Requests;

use App\Models\Lead;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Validation\ValidationException;
use Inertia\Inertia;

class LeadRequest extends FormRequest
{
    private $leadObj;

    /**
     * Creating a new object of Lead model for option values.
     *
     * @return void
     */
    public function __construct(Lead $lead)
    {
        $this->leadObj = $lead;
    }

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $id = $this->route('id');

        return [
            'id' => 'integer|nullable',
            'name' => 'required_without:id|string|max:255',
            'email' => 'required_without:id|string|email|max:255|unique:leads,email,' . $id,
            'phone' => 'required_without:id|numeric',
            'lead_type' => 'required_without:id|string|in:' . implode(',', $this->leadObj->getTypeValues()),
            'lead_source' => 'required_without:id|string|in:' . implode(',', $this->leadObj->getSourceValues()),
            'lead_status' => 'required_without:id|string|in:' . implode(',', $this->leadObj->getStatusValues()),
            'rating' => 'required_without:id|string|in:' . implode(',', $this->leadObj->getRatingValues()),
            'salesperson' => 'required_without:id|integer|exists:users,id',
            'probability' => 'required_without:id|numeric',
            'date_time' => 'required_without:id|string',
            'industry' => 'required_without:id|string',
            'contact_medium' => 'required_without:id|string',
        ];
    }

    public function messages()
    {
        return [
            'id.integer' => 'Invalid lead ID.',
            'name.required_without' => 'Name is required.',
            'name.string' => 'Name should be a string.',
            'name.max' => 'Name should not exceed 255 characters.',
            'email.required_without' => 'Email is required.',
            'email.string' => 'Email should be a string.',
            'email.email' => 'Email is not a valid email.',
            'email.max' => 'Email should not exceed 255 characters.',
            'email.unique' => 'Email already exists.',
            'phone.required_without' => 'Phone is required.',
            'phone.integer' => 'Phone should be an integer.',
            'phone.max' => 'Phone should not exceed 255 characters.',
            'lead_type.required_without' => 'Lead type is required.',
            'lead_type.string' => 'Lead type should be a string.',
            'lead_type.in' => 'Selected lead type is invalid.',
            'lead_source.required_without' => 'Lead source is required.',
            'lead_source.string' => 'Lead source should be a string.',
            'lead_source.in' => 'Selected lead source is invalid.',
            'lead_status.required_without' => 'Lead status is required.',
            'lead_status.string' => 'Lead status should be a string.',
            'lead_status.in' => 'Selected lead status is invalid.',
            'rating.required_without' => 'Rating is required.',
            'rating.string' => 'Rating should be a string.',
            'rating.in' => 'Selected rating is invalid.',
            'salesperson.required_without' => 'Salesperson is required.',
            'salesperson.integer' => 'Salesperson ID should be an integer.',
            'salesperson.exists' => 'Selected salesperson does not exist.',
            'probability.required_without' => 'Probability is required..',
            'date_time.required_without' => 'Date and Time is required.',
            'industry.required_without' => 'Industry is required.',
            'contact_medium.required_without' => 'Contact medium is required.',
        ];
    }

    protected function failedValidation(Validator $validator)
    {
        throw new ValidationException($validator, response()->json([
            'success' => false,
            'message' => $validator->errors()->first(),
            'errors' => $validator->errors()
        ], 422));
    }

    // protected function failedValidation(Validator $validator)
    // {
    //     // dd($validator->errors());
    //     $errors = $validator->errors();

    //     throw new HttpResponseException(
    //         Redirect::back()->withErrors($errors)->withInput()
    //     );

    //     // throw new HttpResponseException(Redirect::back()->with(["errors" => $validator->errors()], 422));
    //     // throw new HttpResponseException(Redirect::route('admin.lead.add')->with(["errors" => $validator->errors()], 422));
    // }

    // protected function failedValidation(Validator $validator)
    // {
    //     throw new ValidationException($validator, Inertia::location(url()->previous())->withErrors($validator)->toResponse(request()));
    // }
}
