<?php

namespace App\Http\Controllers\Payment;

use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\StripePaymentRequest;
use App\Models\CountryCurrencies;
use App\Models\Payment;
use App\Models\PaymentLink;
use App\Models\Setting;
use App\Services\BinApiService;
use App\Services\IpGeoLocationService;
use App\Services\PaymentService;
use App\Services\StripeErrorHandler;
use App\Services\StripeExceptionHandler;
use App\Services\StripeService;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Validator;

use Carbon\Carbon;
use Exception, Session, Stripe;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session as FacadesSession;

class StripeController extends Controller
{
    public $public_key, $secret_key, $statement_descriptor, $environment;
    private $paymentService, $stripeService, $geoLocation, $neutrinoApi;

    public function __construct(PaymentService $paymentService, StripeService $stripeService, IpGeoLocationService $geoLocation, BinApiService $neutrinoApi)
    {
        $this->stripeService = $stripeService;
        $this->paymentService = $paymentService;
        $this->geoLocation = $geoLocation;
        $this->neutrinoApi = $neutrinoApi;
    }

    public function index(Request $request, $token)
    {
        $item_detail = PaymentLink::select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'item_description', 'currency', 'coupon_id', 'sale_type', 'payment_gateway', 'comment')
            ->where("token", "=", $token)
            ->with(
                'currencyCountry:id,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment',
                'paymentLinkCustomer:id,paymentlink_id,name,phone,email,address,company_name'
            )->first();

        $this->public_key = $item_detail->gateway->public_key ?? "";
        $this->secret_key = $item_detail->gateway->secret_key ?? "";
        $this->statement_descriptor = $item_detail->gateway->statement_descriptor ?? "";
        $this->environment = $item_detail->gateway->environment ?? "";

        $countries = CountryCurrencies::select('id', 'aplha_code2', 'aplha_code3', 'country', 'currency', 'code', 'symbol')
            ->orderBy('country', 'ASC')
            ->get();

        Session()->put('payment_gateway', $item_detail->gateway);

        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return view('frontend.payments.pay-page', compact('item_detail', 'countries', 'brand_settings'));
        // 3step
        //return view('frontend.payments.three_step', compact('item_detail', 'countries', 'payment_gateway', 'country_currency'));
    }

    public function createPaymentMethod(StripePaymentRequest $request)
    {
        if ($request->validated()) {
            $formData = (array) $request->all();

            $this->paymentSetting();
            $paymentDetails = (object) $this->itemArray($formData);
            $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            $customerID = 0;
            $payment_method = "";
            $paymentIntent = "";
            $payment = "";

            try {
                $paymentDetails->brand_descriptor = $this->statement_descriptor;
                $paymentDetails->currency_country = $item_detail->currencyCountry;
                $paymentDetails->link_id = $item_detail->id;

                // Initialize Stripe Client with secret Key
                $stripe = new \Stripe\StripeClient($this->secret_key);

                // Creating Customer
                $customer = $this->paymentService->addCustomer($paymentDetails, $item_detail);
                $customerID = array("id" => $customer['internal'], "ip" => $request->ip());

                // Begin Payment Transaction
                $payment = $this->paymentService->beginPayment($paymentDetails, $customerID, $item_detail, $gatewayMessage = "");
                $paymentStripe = $this->paymentService->processPayment($payment, $customer, $item_detail, paymentDetails: $paymentDetails, statement_descriptor: $this->statement_descriptor);

                $paymentResponse = $this->generatePaymentResponse($paymentStripe['payment_intent']);
                $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
                $paymentIntent = $stripe->paymentIntents->retrieve($paymentStripe['payment_intent']->id);
                $location = $this->geoLocation->ipLocation($request->ip());
                $card8Digits = substr($paymentDetails->cardNo, 0, 8);
                $binLookup = $this->neutrinoApi->binLookup($card8Digits, $request->ip());
                $payment_method = $paymentStripe['payment_method'];

                $updatePayment = [
                    'location' => $location->body(),
                    'device' => response()->json($device)->content(),
                    'intent_id' => $paymentIntent->id,
                    'last_four' => $paymentStripe['payment_method']?->card?->last4 ?? 0000,
                    'bin' => response()->json($binLookup)->content(),
                    'intent' => response()->json($paymentIntent)->content(),
                    'paymentMethod' => response()->json($payment_method)->content(),
                ];

                $paymentResponse['payment_id'] = $payment->id;
                $payment->update($updatePayment);

                if ($paymentResponse['success']) {
                    $payment->status = 1;
                    $payment->save();

                    return response()->json($paymentResponse);
                } else if ($paymentResponse['success'] == false && !empty($paymentResponse['requires_action'])) {
                    return response()->json($paymentResponse);
                    //return Redirect::route('payment.stripe.threeDSecure', ['token' => $request->token, 'secret' => $paymentIntent->client_secret, 'pm' => $payment_method->id]);
                } else {
                    $payment->status = 2;
                    $payment->save();
                    return response()->json($paymentResponse);
                    // return Redirect::route('payment.stripe.failed', ['token' => $request->token]);
                }
            } catch (\Exception $e) {
                /* $stripeError = StripeErrorHandler::handle($e, $formData["ptoken"])->content();
        $stripeError = json_decode($stripeError);
        $stripeError->payment_id = $payment->id; */
                return response()->json([
                    'message' => $e->getMessage(),
                    'code' => $e->getCode(),
                    'payment_id' => $payment->id
                ]);
            }
        } else {
            $errors = $request->getMessageBag()->toArray();

            return response()->json([
                "validation_errors" => $errors
            ]);
        }
    }

    public function createThreeStepPaymentMethod(Request $request)
    {
        $formData = Arr::pluck($request->all(), 'value', 'name');

        $formData = array_map(function ($v) {
            return trim(strip_tags($v));
        }, $formData);

        $expDate = $formData["exp-date"];

        $formData["cardNo"] = str_replace(' ', '', $formData["cardNo"]);
        $formData["cardValidity"] = $this->validateCardNumber($formData["cardNo"]);

        if (str_contains($expDate, '/')) {
            $expDate = explode("/", $expDate);
            $formData["month"] = $expDate[0];
            $formData["year"] = $expDate[1];
        } else {
            $formData["month"] = substr($expDate, 0, 2);
            $formData["year"] = substr($expDate, 2, 4);
        }

        $now = Carbon::now();
        $currentYear = $now->format('y');
        $cardNo = str_replace(' ', '', $formData["cardNo"]);
        $rules =  [
            'firstname' => 'required',
            'lastname' => 'required',
            'clientemail' => ['required', 'email'],
            'phonenum' => 'required',
            'address' => 'required',
            'companyname' => 'required',
            'country' => 'required',
            'statename' => 'required',
            'city' => 'required',
            'zipcode' => 'required',
            'cardNo' => ['required', 'digits_between:14,19'],
            'month' => ['required', 'digits:2'],
            'cardValidity' => ['required', function ($attribute, $value, $fail) {
                if (!$value) {
                    $fail("The card number is invalid. Check the card details or use a different card.");
                }
            }],
            'year' => [
                'required', 'digits:2',
                function ($attribute, $value, $fail) use ($currentYear) {
                    if ($currentYear > $value) {
                        $fail("The card's expiration year is incorrect.");
                    }
                },
            ],
            'cvc' => [
                'required', 'digits_between:3,4',
                function ($attribute, $value, $fail) use ($cardNo) {
                    if (strlen((string)$cardNo) === 15 && strlen((string)$value) !== 4) {
                        $fail("Check the card's security code or use a different card.");
                    }
                },
            ],

        ];

        $cardCustomMessages = [
            'cardNo.required' => 'The card number is invalid. ',
            'cardNo.digits_between' => 'The card number is invalid. ',
            'cardValidity.required' => "The card number is invalid. Check the card details or use a different card.",
            'month.required' => "The card's expiration month is incorrect.",
            'month.digits' => "The card's expiration month is incorrect.",
            'year.digits' => "The card's expiration year is incorrect.",
            'year.required' => "The card's expiration year is incorrect.",
            'cvc.required' => "Check the card's security code or use a different card.",
            'cvc.digits_between' => "Check the card's security code or use a different card.",
        ];

        $validator = Validator::make($formData, $rules, $cardCustomMessages);

        if ($validator->fails()) {
            $errors = $validator->getMessageBag()->toArray();

            return response()->json([
                "validation_errors" => $errors
            ]);
        }

        if ($validator->validated()) {
            $this->paymentSetting();

            try {
                $original_price = $formData['itemprice'];
                $currency = (!empty($formData['currency']) ? $formData['currency'] : "USD");
                $converted_amount = $original_price;
                $converted_amount = (!empty($converted_amount) ? $converted_amount : $original_price);
                // 1st transaction
                //$amount = $converted_amount - 1.4;
                $amount = $converted_amount - 1;
                // 2nd transaction
                //$rand = floatVal('0.'.rand(60, 90));
                $rand = 0.5;
                // 3rd transaction
                //$rand2 = $converted_amount - ($amount + $rand);
                $rand2 = 0.5;

                $formData["brand_descriptor"] = $this->statement_descriptor;

                $stripe = Stripe\Stripe::setApiKey($this->secret_key);

                $customer = $this->createCustomer($formData);

                $payment_method = \Stripe\PaymentMethod::create([
                    "billing_details" => [
                        "email" => $formData["clientemail"],
                        "name" => $formData["firstname"] . " " . $formData["lastname"],
                    ],
                    'type' => 'card',
                    'card' => [
                        'number' => $formData["cardNo"],
                        'exp_month' => $formData["month"],
                        'exp_year' => $formData["year"],
                        'cvc' => $formData["cvc"],
                    ],
                ]);

                $payment_method->attach([
                    ['customer' => $customer->id]
                ]);

                unset($formData["cardNo"]);
                unset($formData["month"]);
                unset($formData["year"]);
                unset($formData["cvc"]);

                # Create the PaymentIntent
                $intent = \Stripe\PaymentIntent::create([
                    'payment_method' => $payment_method->id,
                    'amount' => $amount * 100,
                    'currency' => $currency,
                    "customer" => $customer->id,
                    'confirm' => true,
                    'statement_descriptor' => $this->statement_descriptor,
                    'description' => $formData['itemname'],
                    'metadata' => $formData,
                    'automatic_payment_methods' => [
                        'enabled' => true,
                        'allow_redirects' => 'never'
                    ],
                ]);

                return response()->json($intent);
            } catch (\Exception $e) {
                return StripeExceptionHandler::handle($e, $formData["ptoken"]);
            }
        }
    }

    public function PaymentIntent_succeeded(Request $request)
    {
        $this->paymentSetting();
        $json_obj = $request->all();
        $customerData = Arr::pluck($request->paymentData, 'value', 'name');
        $intent = null;

        try {
            if (isset($json_obj['payment_method_id'])) {
                if (isset($customerData['item_desc'])) {
                    unset($customerData['item_desc']);
                }

                if (isset($customerData['payment_gateway'])) {
                    unset($customerData['payment_gateway']);
                }

                $original_price = $customerData['itemprice'];

                $currency = (!empty($customerData['currency']) ? $customerData['currency'] : "USD");
                $converted_amount = $original_price;
                $converted_amount = (!empty($converted_amount) ? $converted_amount : $original_price);

                // 1st transaction
                $amount = $converted_amount - 1;

                // 2nd transaction
                // $rand = floatVal('0.' . rand(60, 90));
                $rand1 = 0.5;
                $rand = $amount - $rand1;
                // 3rd transaction
                // $rand2 = $converted_amount - ($amount + $rand);
                $rand3 = 0.5;
                $rand2 = $rand - $rand3;

                $stripe = Stripe\Stripe::setApiKey($this->secret_key);

                \Stripe\PaymentMethod::all([
                    'customer' => $json_obj['customer_id'],
                    'type' => 'card',
                ]);

                unset($customerData["cardNo"]);
                unset($customerData["month"]);
                unset($customerData["year"]);
                unset($customerData["cvc"]);

                $customer = Stripe\Customer::update(
                    $json_obj['customer_id'],
                    [
                        'description' => $customerData['clientemail'],
                        'email' => $customerData['clientemail'],
                        'phone' => $customerData['phonenum'],
                        'address' => [
                            "line1" => $customerData['address'],
                            "city" => $customerData['city'],
                            "country" => $customerData['country'],
                            "state" => $customerData['statename'],

                        ],
                        'metadata' => $customerData
                    ]
                );

                $intent = \Stripe\PaymentIntent::update(
                    $json_obj['payment_intent_id'],
                    [
                        "customer" => $json_obj['customer_id'],
                        'description' => $customerData['itemname'],
                        'metadata' => $customerData,
                    ]
                );

                if (isset($json_obj['remainingPay']) && $json_obj['remainingPay'] == 'yes') {
                    $charge1 = $this->stripe_charge_paymentIntent($json_obj['customer_id'], $customerData['itemname'], number_format($rand1, 2), "USD");

                    if (isset($charge1->getData()->error)) {
                        return response()->json($intent);
                    } else {
                        $charge2 = $this->stripe_charge_paymentIntent($json_obj['customer_id'], $customerData['itemname'], number_format($rand1, 2), "USD");
                        return response()->json($charge2->getData());
                    }

                    // $transaction1_amount = $charge1->getData()->amount_received/100;
                    // $transaction2_amount = $charge2->getData()->amount_received/100;
                }
            }
        } catch (\Stripe\Exception\CardException $e) {
            // Error code will be authentication_required if authentication is needed
            echo 'Error code is:' . $e->getError()->code;
            $payment_intent_id = $e->getError()->payment_intent->id;
            $payment_intent = \Stripe\PaymentIntent::retrieve($payment_intent_id);
        }
    }

    function paymentIntent_3d(Request $request)
    {
        $this->paymentSetting();
        $customerData = Arr::pluck($request->all(), 'value', 'name');
        $intent = null;

        \Stripe\Stripe::setApiKey($this->secret_key);

        try {
            if (isset($customerData['payment_intent_id'])) {
                $intent = \Stripe\PaymentIntent::retrieve(
                    $customerData['payment_intent_id']
                );
                //$intent->confirm();
            }
            return $this->generatePaymentResponse($intent);
        } catch (Exception $e) {
            return StripeExceptionHandler::handle($e);
        }
    }

    function paymentIntentThreeStep_3d(Request $request)
    {
        $this->paymentSetting();
        $json_obj = $request->all();
        $customerData = Arr::pluck($request->paymentData, 'value', 'name');
        $intent = null;
        $stripe = Stripe\Stripe::setApiKey($this->secret_key);

        try {
            if (isset($json_obj['payment_method_id'])) {
                if (isset($customerData['item_desc'])) {
                    unset($customerData['item_desc']);
                }

                if (isset($customerData['payment_gateway'])) {
                    unset($customerData['payment_gateway']);
                }

                $original_price = $customerData['itemprice'];

                $currency = (!empty($customerData['currency']) ? $customerData['currency'] : "USD");
                $converted_amount = $original_price;
                $converted_amount = (!empty($converted_amount) ? $converted_amount : $original_price);

                //1st transaction
                $amount = $converted_amount - 1;
                // 2nd transaction
                //$rand = floatVal('0.' . rand(60, 90));
                $rand = 0.5;
                // 3rd transaction
                //$rand2 = $converted_amount - ($amount + $rand);
                $rand2 = 0.5;
                $stripe = new \Stripe\StripeClient(
                    $this->secret_key
                );
                $paymentMethods = $stripe->paymentMethods->all([
                    'customer' => $json_obj['customer_id'],
                    'type' => 'card',
                ]);

                unset($customerData["cardNo"]);
                unset($customerData["month"]);
                unset($customerData["year"]);
                unset($customerData["cvc"]);

                $customer = $stripe->customers->update(
                    $json_obj['customer_id'],
                    [
                        'description' => $customerData['clientemail'],
                        'email' => $customerData['clientemail'],
                        'phone' => $customerData['phonenum'],
                        'address' => [
                            "line1" => $customerData['address'],
                            "city" => $customerData['city'],
                            "country" => $customerData['country'],
                            "state" => $customerData['statename'],

                        ],
                        'metadata' => $customerData
                    ]
                );
                $intent = $stripe->paymentIntents->update(
                    $json_obj['payment_intent_id'],
                    [
                        "customer" => $json_obj['customer_id'],
                        'description' => $customerData['itemname'],
                        'metadata' => $customerData,
                    ]
                );

                $stripe->paymentMethods->attach(
                    $json_obj['payment_method_id'],
                    ['customer' => $json_obj['customer_id']]
                );

                $stripe->paymentIntents->confirm(
                    $json_obj['payment_intent_id']
                );


                if (isset($json_obj['remainingPay']) && $json_obj['remainingPay'] == 'yes') {
                    $charge1 = $this->stripe_charge_paymentIntentThreeStep($json_obj['customer_id'], $customerData['itemname'], number_format($rand, 2), "USD");
                    $charge2 = $this->stripe_charge_paymentIntentThreeStep($json_obj['customer_id'], $customerData['itemname'], number_format($rand2, 2), "USD");


                    return response()->json($charge2->getData());


                    $transaction1_amount = $charge1->getData()->amount_received / 100;
                    $transaction2_amount = $charge2->getData()->amount_received / 100;
                }
            }
        } catch (Exception $e) {
            return StripeExceptionHandler::handle($e, $customerData["ptoken"]);
        }
    }

    function generatePaymentResponse($intent)
    {
        # Note that if your API version is before 2019-02-11, 'requires_action'
        # appears as 'requires_source_action'.
        if ($intent->status == 'requires_action' && $intent->next_action->type == 'use_stripe_sdk') {
            # Tell the client to handle the action
            return [
                'success' => false,
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret,
                'payment_method_id' => $intent->payment_method
            ];
        } else if ($intent->status == 'succeeded') {
            # The payment didn't need any additional actions and completed!
            # Handle post-payment fulfillment
            return [
                "success" => true
            ];
        } else {
            # Invalid status
            //http_response_code(500);
            return [
                'success' => false,
                'message' => 'Invalid PaymentIntent status'
            ];
        }
    }

    public function paymentSetting()
    {
        $payment_gateway = Session()->get('payment_gateway');

        $this->public_key = $payment_gateway->public_key;
        $this->secret_key = $payment_gateway->secret_key;
        $this->statement_descriptor = $payment_gateway->statement_descriptor;
        $this->environment = $payment_gateway->environment;
    }

    function validateCardNumber($cardNumber)
    {
        // Remove any non-digit characters from the card number
        $cleanedCardNumber = preg_replace('/\D/', '', $cardNumber);

        // Convert the card number to an array of digits
        $digits = str_split($cleanedCardNumber);

        // Double every second digit starting from the right
        for ($i = count($digits) - 2; $i >= 0; $i -= 2) {
            $digits[$i] *= 2;
            if ($digits[$i] > 9) {
                $digits[$i] -= 9;
            }
        }

        // Sum all the digits
        $sum = array_sum($digits);

        // Check if the sum is divisible by 10
        return $sum % 10 === 0;
    }

    function itemArray($paymentDetails)
    {
        if (is_array($paymentDetails)) {
            return [
                "first_name" => $paymentDetails['firstname'],
                "last_name" => $paymentDetails['lastname'],
                "email" => $paymentDetails['clientemail'],
                "phone" => $paymentDetails['phonenum'],
                "address" => $paymentDetails['address'],
                "company" => $paymentDetails['companyname'],
                "country" => $paymentDetails['country'],
                "state" => $paymentDetails['statename'],
                "city" => $paymentDetails['city'],
                "zipcode" => $paymentDetails['zipcode'],
                "cardNo" => $paymentDetails['cardNo'],
                //"expDate" => $paymentDetails['exp-date'],
                "card_cvc" => $paymentDetails['cvc'],
                "amount" => $paymentDetails['itemprice'],
                "item_name" => $paymentDetails['itemname'],
                "token" => $paymentDetails['ptoken'],
                "discount" => $paymentDetails['discount'],
                "original_amount" => $paymentDetails['original_amount'],
                "gateway" => $paymentDetails['payment_gateway'],
                "currency" => $paymentDetails['currency'],
                "coupon_id" => $paymentDetails['coupon_id'],
                "payment_error" => $paymentDetails['payment_error'],
                "payment_message" => $paymentDetails['payment_message'],
                "payment_response" => $paymentDetails['payment_response'],
                "ThreeDSecure_Popup" => $paymentDetails['3DSecure_Popup'],
                "cardValidity" => $paymentDetails['cardValidity'],
                "cardExp_month" => $paymentDetails['month'],
                "cardExp_year" => $paymentDetails['year']
            ];
        }

        return [];
    }

    // Stripe Payment success
    // After payment process complete it'll redirect to Payment/success or Payment/failed
    public function success(Request $request)
    {
        $formData = Arr::pluck($request->all(), 'value', 'name');
        $formData = (object) array_map(function ($v) {
            return trim(strip_tags($v));
        }, $formData);

        $item_detail = PaymentLink::where("token", "=", $formData->ptoken)
            ->with(
                'currencyCountry:id,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
            )->first();

        $payment = Payment::where("id", "=", $formData->payment_id)->latest('id')->first();


        try {
            if (!empty($formData->payment_decline_code) || !empty($formData->payment_error) || !empty($formData->payment_message)) {
                $errorMessage = $formData->payment_error . '|' . $formData->payment_decline_code . '|' . $formData->payment_message;
                throw new Exception($formData->payment_message, $formData?->payment_error);
            }
            $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
            $location = $this->geoLocation->ipLocation($request->ip());
            $updatePayment = [
                'location' => $location->body(),
                'device' => response()->json($device)->content(),
                'status' => 1
            ];


            \Stripe\Stripe::setApiKey($item_detail->gateway->secret_key);
            Session()->put('payment_gateway', $item_detail->gateway);
            $intent = $this->stripeService->getIntentObject($payment->intent_id);
            $updatePayment['intent'] = response()->json($intent)->content();

            if (isset($intent->latest_charge) && !empty($intent->latest_charge)) {
                $updatePayment['charge_id'] = $intent->latest_charge;
                $updatePayment['charge'] = response()->json($this->stripeService->getChargeObject($intent->latest_charge))->content();
            }

            $payment->update($updatePayment);
            $item_detail->status = 2;
            $item_detail->save();

            if (!empty($formData->threeDSecure_Popup) && $formData->threeDSecure_Popup == 1) {
                $log = array(
                    'activity' => '3D Secure authentication succeeded',
                    'type' => 'threeD_secure',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($updatePayment['intent'])->content(),
                    'code' => response()->json($updatePayment['intent'])->status(),
                );

                $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $log);

                $threeDInfo_log = array(
                    'activity' => 'This payment was verified with 3D Secure and may be protected from being disputed for fraud',
                    'type' => 'info',
                    'request' => [],
                    'response' => [],
                    'code' => 200,
                );

                $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $threeDInfo_log);

                $threeDInfo_log = array(
                    'activity' => 'This payment successfully set up ' . $intent->payment_method?->id . ' for future off-session payments',
                    'type' => 'info',
                    'request' => [],
                    'response' => [],
                    'code' => 200,
                );

                $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $threeDInfo_log);
            }

            $log = array(
                'activity' => 'Payment succeeded',
                'type' => 'success',
                'request' => response()->json($request->all())->content(),
                'response' => $updatePayment['intent'],
                'code' => response()->json($updatePayment['intent'])->status(),
            );

            $paymentLog = $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $log);

            $success_data = array(
                "token" => $formData->ptoken,
                "original_price" => $item_detail->original_price,
                "itemname" => $item_detail->itemname,
                "itemprice" => $item_detail->price,
                "currency" => $item_detail->currencyCountry,
                //"invoice_id" => $invoice_data->invoice_no,
                "payment_id" => $formData->payment_id,
                "discount" => (!empty($item_detail->discount) ? $item_detail->discount : 0),
                "coupon_id" => $item_detail->coupon_id,
            );

            // InvoiceController::email_sent($customer_id);
            InvoiceController::create($payment->id);

            Session()->put("success_data", $success_data);

            if (!empty($paymentLog) && !empty($item_detail->comment)) {
                $commentLog = array(
                    'activity' => $item_detail->comment,
                    'type' => 'note',
                    'request' => '',
                    'response' => '',
                    'code' => 200,
                );

                $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $commentLog);
            }

            NotificationHelper::notify('payment', 'Payment-Received: Amount of' . ' ' . $item_detail->currencyCountry->symbol . $payment->price . ' ' . 'has been received', [
                'id' => $payment->id,
                'price' => $payment->price,
                'message' => 'Payment successful',
            ], $item_detail->created_by);

            return response()->json(['success' => true, 'route' => route('payment.success', ["token" => $formData->ptoken])], 200);
        } catch (Exception $ex) {
            $stripeError = explode('|', $ex->getMessage());
            $log = array(
                'activity' => 'Payment Failed: ' . (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0]),
                'type' => 'failed',
                'request' => response()->json($formData)->content(),
                'response' => response()->json(['code' => (!empty($stripeError[1]) ? $stripeError[1] : $ex->getCode()), 'message' => (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0])])->content(),
                'code' => $ex->getCode(),
            );

            $paymentLog = $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $log);

            $payment->update([
                'status' => 2,
            ]);

            report($ex);
            Log::error("Token: " . $formData->ptoken . " has an exception on Storing Payment. Message: " . (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0]));

            $failed_data = array(
                "token" => $formData->ptoken,
                'current_time' => Carbon::now()->timestamp,
                'currency' => $formData->currency,
                'itemprice' => $formData->itemprice,
                'itemname' => $formData->itemname,
                'title' => 'Payment Failed | ' . $formData->currency . $formData->itemname . ' | Message: ' . (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0]),
                'heading' => 'Payment Failed',
                'code' => (!empty($stripeError[1]) ? $stripeError[1] : $ex->getCode()),
                'message' => (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0])
            );

            Session()->put("failed_data", $failed_data);

            if (!empty($paymentLog) && !empty($item_detail->comment)) {
                $commentLog = array(
                    'activity' => $item_detail->comment,
                    'type' => 'note',
                    'request' => '',
                    'response' => '',
                    'code' => 200,
                );

                $this->paymentService->log($item_detail->created_by, Payment::class, $payment->id, $commentLog);
            }

            NotificationHelper::notify('payment-failed', 'Payment-Failed: Amount of $' . $payment->price . ' ' . 'has been failed', [
                'id' => $payment->id,
                'price' => $payment->price,
                'request' => response()->json($formData)->content(),
                'response' => response()->json(['code' => (!empty($stripeError[1]) ? $stripeError[1] : $ex->getCode()), 'message' => (!empty($stripeError[2]) ? $stripeError[2] : $stripeError[0])])->content(),
                'code' => $ex->getCode(),
            ], $item_detail->created_by);

            return response()->json(['failed' => true, 'route' => route('payment.failed', ["token" => $formData->ptoken])], 200);
        }
    }
}
