<?php

namespace App\Http\Controllers\Payment;

use App\Exceptions\StripeException;
use App\Helper\Helper\NotificationHelper;
use App\Helper\Helper\StripeHelper;
use App\Helper\PaymentHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\PaymentRequest;
use App\Jobs\ApiLogJob;
use App\Models\Coupon;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\PaymentLink;
use App\Models\User;
use App\Repositories\Payment\Errors\ErrorsRepository;
use App\Repositories\Payment\PaymentRepository;
use App\Services\BinApiService;
use App\Services\CurrencyService;
use App\Services\IpGeoLocationService;
use App\Services\LogService;
use App\Services\Payment\PaymentService;
use App\Services\Payment\Stripe\Charge;
use App\Services\Payment\Stripe\PaymentIntent;
use App\Services\Payment\Stripe\PaymentMethod;
use App\Services\SettingService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Hash;

class PaynowController extends Controller
{
    private $public_key, $secret_key, $statement_descriptor, $environment;
    private $paymentService, $geoLocation, $neutrinoApi, $logRepository, $settingService, $currencyService, $baseUrl, $clientId, $clientSecret;

    public function __construct(IpGeoLocationService $geoLocation, BinApiService $neutrinoApi, LogService $log, SettingService $settingService, CurrencyService $currencyService)
    {
        $this->geoLocation = $geoLocation;
        $this->neutrinoApi = $neutrinoApi;
        $this->logRepository = $log;
        $this->settingService = $settingService;
        $this->currencyService = $currencyService;
        $this->baseUrl = config('services.paypal.mode') === 'sandbox'
            ? 'https://api-m.sandbox.paypal.com'
            : 'https://api-m.paypal.com';
        $this->clientId = config('services.paypal.client_id');
        $this->clientSecret = config('services.paypal.client_secret');
    }

    public function index(Request $request, $token)
    {
        $item_detail = PaymentLink::select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'item_description', 'currency', 'coupon_id', 'show_coupon', 'sale_type', 'payment_gateway', 'comment', 'created_by')
            ->where("token", "=", $token)
            ->with(
                'currencyCountry:id,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment',
                'paymentLinkCustomer:id,paymentlink_id,name,phone,email,address,company_name'
            )->first();

        $this->public_key = $item_detail->gateway->public_key ?? "";
        $this->secret_key = $item_detail->gateway->secret_key ?? "";
        $this->statement_descriptor = $item_detail->gateway->statement_descriptor ?? "";
        $this->environment = $item_detail->gateway->environment ?? "";

        Session()->put('payment_gateway', $item_detail->gateway);

        return Inertia::render('Payment/Checkout', [
            'countries' => $this->currencyService->getCountriesCurrency(),
            'item' => $item_detail,
            'brand_settings' => $this->settingService->get()
        ]);
    }

    // index method for paypal checkout
    public function indexPaypalCheckout(Request $request, $token)
    {
        $item_detail = PaymentLink::with(
            'currencyCountry:id,aplha_code3,code,symbol',
            'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment',
            'paymentLinkCustomer:id,paymentlink_id,name,phone,email,address,company_name'
        )->where("token", $token)->first();

        if (!$item_detail) {
            return redirect()->back()->with('error', 'Invalid payment link token.');
        }

        $this->public_key = $item_detail->gateway->public_key ?? "";
        $this->secret_key = $item_detail->gateway->secret_key ?? "";
        $this->statement_descriptor = $item_detail->gateway->statement_descriptor ?? "";
        $this->environment = $item_detail->gateway->environment ?? "";

        Session()->put('payment_gateway', $item_detail->gateway);

        return Inertia::render('Payment/CheckoutPaypal', [
            'countries' => $this->currencyService->getCountriesCurrency(),
            'item' => $item_detail,
            'brand_settings' => $this->settingService->get()
        ]);
    }

    public function paymentSetting()
    {
        $payment_gateway = Session()->get('payment_gateway');

        $this->public_key = $payment_gateway->public_key;
        $this->secret_key = $payment_gateway->secret_key;
        $this->statement_descriptor = $payment_gateway->statement_descriptor;
        $this->environment = $payment_gateway->environment;
    }

    public function createCustomer(PaymentRequest $request, PaymentService $paymentService)
    {
        $formData = (array) $request->all();
        $paymentDetails = (object) PaymentHelper::itemArray($formData);
        $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
            ->with(
                'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,statement_descriptor,environment'
            )->first();

        $customerID = 0;
        $payment_method = "";
        $paymentIntent = "";

        if ($request->payment_id) {
            $payment = (object) ["id" => $request->payment_id];
        } else {
            $payment = "";
        }

        try {
            if ($request->error) {
                $error = $request->error;
                throw new StripeException($error['message'], $error['type'], $error['statusCode'], '', $error['statusCode'], $error);
            }

            if ($paymentDetails->coupon_id > 0) {
                $coupon = Coupon::select('id', 'discount', 'discount_type')->where('id', $paymentDetails->coupon_id)->first();
                $discount = PaymentHelper::claimCoupon($item_detail->price, $coupon->discount, $coupon->discount_type);

                $item_detail->coupon_id = $coupon->id;
                $item_detail->price = $discount['amount'];
                $item_detail->discount = $discount['discount'];
                $item_detail->discount_type = $discount['discountType'];
                $item_detail->save();

                $item_detail->refresh();
            }

            $paymentDetails->brand_descriptor = $this->statement_descriptor;
            $paymentDetails->currency_country = $item_detail->currencyCountry;
            $paymentDetails->link_id = $item_detail->id;

            $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
            $location = $this->geoLocation->ipLocation($request->ip());
            $geoLocate = $this->neutrinoApi->geolocate(address: $paymentDetails->address, houseNumber: "", street: "", city: $paymentDetails->city, county: $paymentDetails->address, state: $paymentDetails->state, postalCode: $paymentDetails->zipcode, countryCode: "", languageCode: 'en', fuzzySearch: false);
            $geoAddress['geolocation'] = !empty($geoLocate['locations']) ? $geoLocate['locations'] : '';
            $locationResponse = !empty($location->body()) ? array_merge(json_decode($location->body(), true), $geoAddress) : $geoAddress;

            $updatePayment = [
                'location' => response()->json($locationResponse)->content(),
                'device' => response()->json($device)->content()
            ];

            if ($request->card_type == 'creditCard') {
                $card8Digits = substr($paymentDetails->cardNo, 0, 8);
                $paymentDetails->cardNo = StripeHelper::maskNumber($paymentDetails->cardNo);

                StripeHelper::removeKey($paymentDetails, 'card_number');
                StripeHelper::removeKey($paymentDetails, 'card_cvc');
                StripeHelper::removeKey($paymentDetails, 'card_date');

                $binLookup = $this->neutrinoApi->binLookup($card8Digits, $request->ip());
                $updatePayment['bin'] = response()->json($binLookup)->content();
            }

            // Creating Customer
            $customer = $paymentService->addCustomer($paymentDetails, $item_detail);

            // Begin Payment Transaction
            $payment = $paymentService->initiate($customer['internal'], $item_detail, $request->ip(), $item_detail->comment, "");

            $delay = Carbon::now()->addMinutes(2);
            $this->queueApiLog('NEUTRINO UA-Lookup', ['user-agent' => $request->header('User-Agent')], $updatePayment['device'], $payment->id, $item_detail->created_by, $delay);
            $this->queueApiLog('NEUTRINO GeoAddress', ['ip' => $request->ip()], $updatePayment['location'], $payment->id, $item_detail->created_by, $delay);

            if ($request->card_type == 'creditCard') {
                $this->queueApiLog('NEUTRINO Bin-Lookup', ['bin' => $card8Digits], $updatePayment['bin'], $payment->id, $item_detail->created_by, $delay);
            }

            $updatePayment['intent'] = response()->json($paymentIntent)->content();
            $updatePayment['paymentMethod'] = response()->json($payment_method)->content();

            $paymentResponse['success'] = true;
            $paymentResponse['data'] = array(
                'payment_id' => $payment->id,
                'customer_id' => $customer['internal'],
                'stripe' => array(
                    'customer_id' => $customer['customer']->id
                )
            );

            $payment->update($updatePayment);
            $payment->save();

            return response()->json($paymentResponse);
        } catch (StripeException $e) {
            // log error
            $this->logRepository->log('payment', [
                'activity' => 'Payment Failed because of ' . (!empty($e->getError()->decline_code) ? $e->getError()->type . ' ' . $e->getError()->code . ' | ' . $e->getError()->decline_code . ' | ' . $e->getError()->message : $e->getMessage()),
                'type' => 'failed',
                'request' => response()->json($paymentDetails)->content(),
                'response' => response()->json($e->getError())->content(),
                'code' => $e->getHttpStatusCode(),
                'loggable_id' => $payment->id,
                'created_by' => $item_detail->created_by,
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'payment_id' => $payment->id,
                'error' => $e->getError()
            ], $e->getHttpStatusCode());
        }
    }

    public function createPaymentMethod(PaymentRequest $request, PaymentService $paymentService, Charge $stripeCharge, PaymentIntent $stripePaymentIntent)
    {
        $formData = (array) $request->all();

        $this->paymentSetting();
        $paymentDetails = (object) PaymentHelper::itemArray($formData);
        $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
            ->with(
                'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
            )->first();

        $customer = [
            'stripe_id' => $request->stripe_customer_id,
            'internal' => $request->customer_id
        ];
        $payment_method = "";
        $paymentIntent = "";
        $payment = Payment::find($request->payment_id);

        try {
            $this->statement_descriptor = $item_detail->gateway->statement_descriptor;
            $paymentDetails->brand_descriptor = $this->statement_descriptor;
            $paymentDetails->currency_country = $item_detail->currencyCountry;
            $paymentDetails->link_id = $item_detail->id;

            $initiatePayment = $paymentService->process($payment, $customer, $item_detail, $paymentDetails, $this->statement_descriptor, $item_detail->comment);

            $paymentResponse = $this->generatePaymentResponse($initiatePayment['payment_intent']);

            $paymentIntent = $stripePaymentIntent->get($initiatePayment['payment_intent']->id);
            $payment_method = $initiatePayment['payment_method'];

            $updatePayment = [
                'intent_id' => $paymentIntent->id,
                'last_four' => $initiatePayment['payment_method']?->card?->last4 ?? 0000,
                'intent' => response()->json($paymentIntent)->content(),
                'paymentMethod' => response()->json($payment_method)->content(),
            ];

            $paymentResponse['payment_id'] = $payment->id;
            $payment->update($updatePayment);

            if ($paymentResponse['success']) {
                $charge = $stripeCharge->get($paymentIntent->latest_charge);
                $payment->update([
                    'charge_id' => $charge->id,
                    'charge' => response()->json($charge)->content()
                ]);

                $this->logRepository->log('payment', [
                    'activity' => 'Payment succeeded',
                    'type' => 'success',
                    'request' => response()->json($paymentDetails)->content(),
                    'response' => response()->json($charge)->content(),
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ]);

                $payment->status = 1;
                $payment->save();
            } else if ($paymentResponse['success'] == false && !empty($paymentResponse['requires_action'])) {
                $this->logRepository->log('payment', [
                    'activity' => 'Payment requires action',
                    'type' => 'payment.intent.confirmation',
                    'request' => response()->json([])->content(),
                    'response' => response()->json($paymentIntent)->content(),
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ], "", 1);

                $this->logRepository->eventStatus($paymentIntent->status, $payment->id, response()->json($paymentIntent)->content(), $item_detail->created_by);

                return response()->json($paymentResponse);
            } else {
                $payment->status = 2;
                $payment->save();
            }

            return response()->json($paymentResponse);
        } catch (StripeException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'payment_id' => $payment->id,
                'error' => $e->getError()
            ], $e->getHttpStatusCode());
        }
    }

    public function chargeAfterPay(PaymentRequest $request, PaymentService $paymentService, Charge $stripeCharge, PaymentIntent $stripePaymentIntent)
    {
        $formData = (array) $request->all();

        $this->paymentSetting();
        $paymentDetails = (object) PaymentHelper::itemArray($formData);
        $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
            ->with(
                'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
            )->first();

        $customer = [
            'stripe_id' => $request->stripe_customer_id,
            'internal' => $request->customer_id
        ];
        $payment_method = "";
        $paymentIntent = "";
        $payment = Payment::find($request->payment_id);

        try {
            $this->statement_descriptor = $item_detail->gateway->statement_descriptor;
            $paymentDetails->brand_descriptor = $this->statement_descriptor;
            $paymentDetails->currency_country = $item_detail->currencyCountry;
            $paymentDetails->link_id = $item_detail->id;

            $initiatePayment = $paymentService->afterpay_process($payment, $customer, $item_detail, $paymentDetails, $this->statement_descriptor, $item_detail->comment);
            $paymentResponse = $this->generatePaymentResponse($initiatePayment['payment_intent']);

            $paymentIntent = $stripePaymentIntent->get($initiatePayment['payment_intent']->id);

            $updatePayment = [
                'intent_id' => $paymentIntent->id,
                'intent' => response()->json($paymentIntent)->content(),
            ];

            $paymentResponse['payment_id'] = $payment->id;
            $payment->update($updatePayment);

            if ($paymentResponse['success']) {
                $charge = $stripeCharge->get($paymentIntent->latest_charge);
                $payment->update([
                    'charge_id' => $charge->id,
                    'charge' => response()->json($charge)->content()
                ]);

                $this->logRepository->log('payment', [
                    'activity' => 'Payment succeeded',
                    'type' => 'success',
                    'request' => response()->json($paymentDetails)->content(),
                    'response' => response()->json($charge)->content(),
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ]);

                $payment->status = 1;
                $payment->save();
            } else if ($paymentResponse['success'] == false && !empty($paymentResponse['requires_action'])) {
                return response()->json($paymentResponse);
            } else {
                $payment->status = 2;
                $payment->save();
            }

            return response()->json($paymentResponse);
        } catch (StripeException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'payment_id' => $payment->id,
                'error' => $e->getError()
            ], $e->getHttpStatusCode());
        }
    }

    public function successAfterPay(Request $request, PaymentIntent $stripePaymentIntent, Charge $stripeCharge, PaymentMethod $stripePaymentMethod, ErrorsRepository $errorsRepository)
    {
        $payment = '';
        $item_detail = '';

        try {
            $payment = Payment::find($request->payment_id);
            $item_detail = PaymentLink::where("id", "=", $payment->payment_link_id)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            $paymentIntent = $stripePaymentIntent->get($request->payment_intent);
            $paymentResponse = $this->generatePaymentResponse($paymentIntent);

            if ($paymentResponse['success']) {
                InvoiceController::create($payment->id, false, 1);

                $success_data = array(
                    "token" => $item_detail->token,
                    "original_price" => $item_detail->original_price,
                    "item_name" => $item_detail->item_name,
                    "price" => $item_detail->price,
                    "currency_country" => $item_detail->currencyCountry,
                    // "invoice_id" => $invoice_data->invoice_no,
                    "payment_id" => $payment->id,
                    "discount" => (!empty($formData['discount']) ? $formData['discount'] : 0),
                );

                if (!empty($item_detail->comment)) {
                    $commentLog = array(
                        'activity' => $item_detail->comment,
                        'type' => 'note',
                        'request' => '',
                        'response' => '',
                        'code' => 200,
                        'loggable_id' => $payment->id,
                        'created_by' => $item_detail->created_by,
                    );

                    $this->logRepository->log('payment', $commentLog);
                }

                $charge = $stripeCharge->get($paymentIntent->latest_charge);
                $paymentMethod = $stripePaymentMethod->get($paymentIntent->payment_method);

                $payment->update([
                    'intent_id' => $paymentIntent->id,
                    'intent' => response()->json($paymentIntent)->content(),
                    'charge_id' => $charge->id,
                    'charge' => response()->json($charge)->content(),
                    'paymentMethod' => response()->json($paymentMethod)->content()
                ]);

                $this->logRepository->log('payment', [
                    'activity' => 'Payment succeeded',
                    'type' => 'success',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($charge)->content(),
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ]);

                $payment->status = 1;
                $payment->save();

                User::where('id', $payment->customer_id)->update([
                    'stripe_pm_id' => $paymentMethod->id,
                    'stripe_payment_method' => response()->json($paymentMethod)->content()
                ]);

                NotificationHelper::notify('payment', 'Payment-Received: Amount of ' . $item_detail->currencyCountry->symbol . $payment->price . ' ' . 'has been received', [
                    'id' => $payment->id,
                    'price' => $payment->price,
                    'message' => 'Payment successful',
                ], $item_detail->created_by);

                Session::flash("success_data", $success_data);

                $item_detail->payment_gateway = 3;
                $item_detail->status = 2;
                $item_detail->save();

                return Redirect::route('paynow.success', ["token" => $item_detail->token]);
            } else {
                if (!empty($paymentIntent['last_payment_error'])) {
                    $error = $paymentIntent['last_payment_error'];
                    $errorMessage = $errorsRepository->getMessage($error['code']);

                    if (!$errorMessage) {
                        $errorMessage = $errorsRepository->getMessage('generic_decline');
                    }

                    $this->logRepository->log('payment', [
                        'activity' => 'Payment Failed because of ' . (!empty($error['code']) ? $error['code'] . ' | ' . $error['message'] : $error['message']),
                        'type' => 'failed',
                        'request' => response()->json($request->all())->content(),
                        'response' => response()->json($error)->content(),
                        'code' => 402,
                        'loggable_id' => $payment->id,
                        'created_by' => $item_detail->created_by,
                    ]);

                    if ($payment->status == 3) {
                        $payment->status = 2;
                        $payment->save();
                    }

                    $payment->gateway_message = response()->json($errorMessage)->content();
                    $payment->save();
                }

                $chatAgentMessage = (!empty($errorMessage["chat_code"]) ? $errorMessage["chat_code"] : $errorMessage["message"]);
                $sessional_time = Carbon::now()->addMinutes(10)->timestamp;

                $failed_data = array(
                    "token" => $item_detail->token,
                    'current_time' => Carbon::now()->timestamp,
                    'session' => $sessional_time,
                    'currency_country' => $item_detail->currencyCountry,
                    'price' => $item_detail->price,
                    'item_name' => $item_detail->item_name,
                    'title' => 'Payment Failed | ' . $item_detail->item_name . ' (' . $item_detail->currencyCountry->code . $item_detail->price . ') | Message: ' . $chatAgentMessage,
                    'heading' => 'Payment Failed',
                    'message' => [
                        'title' => $errorMessage['stripe_message'],
                        'code' => $errorMessage["code"],
                        'chat_agent' => $chatAgentMessage,
                        'customer' => $errorMessage["message"]
                    ]
                );

                NotificationHelper::notify('payment-failed', 'Payment-Failed: Amount of ' . $item_detail->currencyCountry->code . $payment->price . ' ' . 'has been failed', [
                    'id' => $payment->id,
                    'price' => $payment->price,
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($error)->content(),
                    'code' => 402
                ], $item_detail->created_by);

                Session::flash("failed_data", $failed_data);
                // Reflash the flash data for 3 subsequent requests
                Session::reflash();

                return Redirect::route('paynow.failed', ["token" => $item_detail->token]);
            }
        } catch (StripeException $e) {
            if ($payment) {
                $payment->status = 2;
                $payment->save();
            } else {
                $payment = Payment::where('id', $request->payment_id)->update([
                    'status' => 2
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'payment_id' => $payment->id,
                'error' => $e->getError()
            ], $e->getHttpStatusCode());
        }
    }

    function generatePaymentResponse($intent)
    {
        # Note that if your API version is before 2019-02-11, 'requires_action'
        # appears as 'requires_source_action'.
        if ($intent->status == 'requires_action' && $intent->next_action->type == 'use_stripe_sdk') {
            # Tell the client to handle the action
            return [
                'success' => false,
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret,
                'payment_method_id' => $intent->payment_method
            ];
        } else if ($intent->status == 'requires_payment_method') {
            return [
                "success" => false,
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret,
            ];
        } else if ($intent->status == 'succeeded') {
            # The payment didn't need any additional actions and completed!
            # Handle post-payment fulfillment
            return [
                "success" => true
            ];
        } else {
            # Invalid status
            //http_response_code(500);
            return [
                'success' => false,
                'message' => 'Invalid PaymentIntent status'
            ];
        }
    }

    function verify3DSecure(Request $request, PaymentIntent $paymentIntent, PaymentRepository $payment, Charge $stripeCharge)
    {
        $paymentIntentID = $request->payment_intent_id;
        $paymentID = $request->payment_id;
        $userID = $request->userID;

        try {
            $paymentIntentObj = $paymentIntent->get($paymentIntentID);

            if ($paymentIntentObj?->latest_charge) {
                $charge = $stripeCharge->get($paymentIntentObj->latest_charge);
            } else {
                $charge = (object) [];
            }

            $payment->update($paymentID, [
                'intent_id' => $paymentIntentObj->id,
                'intent' => response()->json($paymentIntentObj)->content(),
                'charge_id' => $charge->id ?? '',
                'charge' => response()->json($charge)->content()
            ]);

            $response = $this->generatePaymentResponse($paymentIntentObj);

            if ($response['success']) {
                $this->logRepository->log('payment', [
                    'activity' => '3D Secure authentication succeeded',
                    'type' => 'threeD_secure',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($paymentIntentObj)->content(),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $userID,
                ]);

                Payment::where('id', $paymentID)->update([
                    'status' => 1
                ]);
            }
            return $response;
        } catch (StripeException $e) {
            // log error
            $this->logRepository->log('payment', [
                'activity' => 'ThreeD Secure Authentication failed: ' . (!empty($e->getError()->decline_code) ? 'type: ' . $e->getError()->type . ' ' . $e->getError()->code . ':' . $e->getError()->decline_code : $e->getMessage()),
                'type' => 'card.failed',
                'request' => response()->json(['payment_intent_id' => $paymentIntentID, 'payment_id' => $paymentID]),
                'response' => response()->json($e->getError())->content(),
                'code' => $e->getHttpStatusCode(),
                'loggable_id' => $paymentID,
                'created_by' => $userID,
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'payment_id' => $paymentID,
                'error' => $e->getError()
            ], 402);
        }
    }

    // For retry payment when payment failed
    public function previousData($token)
    {
        $tokenData = StripeHelper::readFile($token);

        $item_detail = PaymentLink::select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'item_description', 'currency', 'coupon_id', 'sale_type', 'payment_gateway', 'comment', 'created_by')
            ->where("token", "=", $token)
            ->with(
                'currencyCountry:id,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment',
                'paymentLinkCustomer:id,paymentlink_id,name,phone,email,address'
            )->first();

        $this->public_key = $item_detail->gateway->public_key ?? "";
        $this->secret_key = $item_detail->gateway->secret_key ?? "";
        $this->statement_descriptor = $item_detail->gateway->statement_descriptor ?? "";
        $this->environment = $item_detail->gateway->environment ?? "";

        Session()->put('payment_gateway', $item_detail->gateway);

        if ($tokenData['data']['gateway'] == '7' || $tokenData['data']['gateway'] == '8') {

            $comonentName = 'Payment/CheckoutPaypal';
        } else {
            $comonentName = 'Payment/Checkout';
        }
        // dd($tokenData, $item_detail, );
        return Inertia::render($comonentName, [
            'countries' => $this->currencyService->getCountriesCurrency(),
            'item' => $item_detail,
            'brand_settings' => $this->settingService->get(),
            'tokenData' => $tokenData
        ]);
    }

    // finalizing the payment adding details in database and queuing email and notification
    public function verifyPayment(PaymentRequest $request, ErrorsRepository $errorsRepository, PaymentService $paymentService)
    {
        $sessional_time = Carbon::now()->addMinutes(10)->timestamp;
        $formData = (array) $request->all();
        $paymentDetails = (object) PaymentHelper::itemArray($formData);

        $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
            ->with(
                'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
            )->first();

        $error = $request->error;
        $message = $request->messsage;
        $success = $request->success;

        $payment = Payment::find($request->payment_id);

        try {
            if (!empty($error['code']) || !empty($error['type'])) {
                throw new StripeException($error['message'], $error['type'], 402, '', 402, $error);
            }

            InvoiceController::create($payment->id, false, 1);
            // $invoice_data = Invoice::where('payment_id', $payment->id)->first();

            $success_data = array(
                "token" => $item_detail->token,
                "original_price" => $item_detail->original_price,
                "item_name" => $item_detail->item_name,
                "price" => $item_detail->price,
                "currency_country" => $item_detail->currencyCountry,
                // "invoice_id" => $invoice_data->invoice_no,
                "payment_id" => $payment->id,
                "discount" => (!empty($formData['discount']) ? $formData['discount'] : 0),
                "coupon_id" => $formData['coupon_id'],
            );

            // InvoiceController::email_sent($customer_id);

            if (!empty($item_detail->comment)) {
                $commentLog = array(
                    'activity' => $item_detail->comment,
                    'type' => 'note',
                    'request' => '',
                    'response' => '',
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                );

                $this->logRepository->log('payment', $commentLog);
            }

            $paymentService->stripeChargeDetails($payment->id, $payment->intent_id, $payment->customer_id, $item_detail->created_by);

            if (str_contains($item_detail->gateway->gateway, 'three_step')) {
                $customer = User::where('id', $payment->customer_id)->first();
                $paymentService->threestep_point_transaction($payment, $item_detail, $customer);
                $paymentService->threestep_point_transaction($payment, $item_detail, $customer);
            }

            NotificationHelper::notify('payment', 'Payment-Received: Amount of ' . $item_detail->currencyCountry->symbol . $payment->price . ' ' . 'has been received', [
                'id' => $payment->id,
                'price' => $payment->price,
                'message' => 'Payment successful',
            ], $item_detail->created_by);

            Session::flash("success_data", $success_data);

            $item_detail->status = 2;
            $item_detail->save();

            // Creating encrypted file for retry customer
            $paymentService->generateFile($paymentDetails->token, $paymentDetails, $payment->id, $item_detail->created_by);


            return response()->json(['success' => true, 'route' => route('paynow.success', ["token" => $item_detail->token])], 200);
        } catch (StripeException $e) {
            $code = '';
            $errorMessage = '';

            if (!empty($e->getError()['decline_code']) || !empty($e->getError()['code'])) {
                $code = !empty($e->getError()['decline_code']) ? $e->getError()['decline_code'] : (!empty($e->getError()['code']) ? $e->getError()['code'] : '');

                if (!empty($e->getError()['decline_code']) && $e->getError()['decline_code'] == 'generic_decline') {
                    $errorMessage = $errorsRepository->getMessage($e->getError()['code']);
                } else {
                    $errorMessage = $errorsRepository->getMessage($code);
                }
            }
            if (!$errorMessage) {
                $errorMessage = $errorsRepository->getMessage('generic_decline');
            }

            if (!empty($error)) {
                $this->logRepository->log('payment', [
                    'activity' => 'Payment Failed because of ' . (!empty($e->getError()->code) ? $e->getError()->code . ' | ' . $e->getError()->message : $e->getMessage()),
                    'type' => 'failed',
                    'request' => response()->json($paymentDetails)->content(),
                    'response' => response()->json($e->getError())->content(),
                    'code' => $e->getHttpStatusCode(),
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ]);

                if (!empty($e->getError()['code']) && $e->getError()['code'] == 'payment_intent_authentication_failure') {
                    $payment->status = 5;
                    $payment->save();
                }
            }

            if ($payment->status == 3) {
                $payment->status = 2;
                $payment->save();
            }

            $payment->gateway_message = response()->json($errorMessage)->content();
            $payment->save();

            $chatAgentMessage = (!empty($errorMessage["chat_code"]) ? $errorMessage["chat_code"] : $errorMessage["message"]);

            $failed_data = array(
                "token" => $item_detail->token,
                'current_time' => Carbon::now()->timestamp,
                'session' => $sessional_time,
                'currency_country' => $item_detail->currencyCountry,
                'price' => $item_detail->price,
                'item_name' => $formData['itemname'],
                'title' => 'Payment Failed | ' . $formData['itemname'] . ' (' . $item_detail->currencyCountry->code . $item_detail->price . ') | Message: ' . $chatAgentMessage,
                'heading' => 'Payment Failed',
                'message' => [
                    'title' => $errorMessage['stripe_message'],
                    'code' => $errorMessage["code"],
                    'chat_agent' => $chatAgentMessage,
                    'customer' => $errorMessage["message"]
                ]
            );

            NotificationHelper::notify('payment-failed', 'Payment-Failed: Amount of ' . $item_detail->currencyCountry->code . $payment->price . ' ' . 'has been failed', [
                'id' => $payment->id,
                'price' => $payment->price,
                'request' => response()->json($formData)->content(),
                'response' => response()->json(['code' => $e->getCode(), 'message' => $e->getMessage()])->content(),
                'code' => $e->getCode(),
            ], $item_detail->created_by);

            // Creating encrypted file for retry customer
            $paymentService->generateFile($paymentDetails->token, $paymentDetails, $payment->id, $item_detail->created_by);

            Session::flash("failed_data", $failed_data);
            // Reflash the flash data for 3 subsequent requests
            Session::reflash();
            return response()->json(['success' => false, 'route' => route('paynow.failed', ["token" => $item_detail->token])], 200);
        }
    }

    public function success(Request $request)
    {
        if (Session()->get('success_data')) {
            $data = Session()->get('success_data');
            $payment = PaymentLink::select('id', 'token')->where('token', $data["token"])
                ->with([
                    'payments' => function ($payment) use ($data) {
                        if (!empty($data['payment_id'])) {
                            $payment->where('id', $data['payment_id']);
                        }
                        $payment->orderBy('id', 'ASC');
                        $payment->limit(1);
                        $payment->with('invoice');
                        $payment->with('customer');
                    }
                ]);

            if (env('IS_CATEGORY_ENABLED')) {
                $payment = $payment->with('categories');
            }

            $payment = $payment->first();

            Session()->forget('success_data');

            $saleType = $payment->sale_type;
            $invoice_no = $payment->payments->invoice->invoice_no;
            $categories = $payment->categories;
            $categoryNames = $categories->pluck('name')->implode(' + ');

            $script = "<script async src='https://www.googletagmanager.com/gtag/js?id=" . env('GTAG_SCRIPT_KEY') . "'></script><script> window.dataLayer = window.dataLayer || []; function gtag(){dataLayer.push(arguments);} gtag('js', new Date()); gtag('config', '" . env('GTAG_SCRIPT_KEY') . "'); </script>";
            $logoDesign = '<script> gtag("event", "purchase", { "send_to": "' . env('GTAG_SCRIPT_KEY') . '","transaction_id": "' . $invoice_no . '","affiliation": "' . $saleType . '","value": ' . $payment->price . ', "currency": "USD","items": [{"id": "' . $invoice_no . '", "name": "' . $payment->item_name . '","category": "' . $categoryNames . '","quantity": 1, "price": "' . $payment->price . '"},]});</script>';
            $gtag_script = $script . $logoDesign;

            return Inertia::render('Payment/Success', [
                "data" => $data,
                "payment" => $payment,
                'brand_settings' => $this->settingService->get(),
                'gtag_script' => $gtag_script
            ]);


            return Inertia::render('Payment/Success', [
                "data" => $data,
                "payment" => $payment,
                'brand_settings' => $this->settingService->get()
            ]);
        } else {
            return redirect()->route('payment.expired');
        }
    }

    private function generateAccessToken()
    {
        $client = new Client();
        $response = $client->post("{$this->baseUrl}/v1/oauth2/token", [
            'auth' => [$this->clientId, $this->clientSecret],
            'form_params' => [
                'grant_type' => 'client_credentials',
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['access_token'];
    }

    // redirected to success paypal
    public function successByPaypal(Request $request, PaymentService $paymentService)
    {
        if (Session()->get('success_paypal_data')) {
            $payment = PaymentLink::select('id', 'token')->where('token', Session()->get('success_paypal_data')["token"])
                ->with([
                    'payments' => function ($payment) {
                        $payment->orderBy('id', 'ASC');
                        $payment->limit(1);
                        $payment->with('invoice');
                        $payment->with('customer');
                    }
                ]);

            if (env('IS_CATEGORY_ENABLED')) {
                $payment = $payment->with('categories');
            }

            $payment = $payment->first();

            $saleType = $payment->sale_type;
            $invoice_no = $payment->payments->invoice->invoice_no;
            $categories = $payment->categories;
            $categoryNames = $categories->pluck('name')->implode(' + ');

            $script = "<script async src='https://www.googletagmanager.com/gtag/js?id=" . env('GTAG_SCRIPT_KEY') . "'></script><script> window.dataLayer = window.dataLayer || []; function gtag(){dataLayer.push(arguments);} gtag('js', new Date()); gtag('config', '" . env('GTAG_SCRIPT_KEY') . "'); </script>";
            $logoDesign = '<script> gtag("event", "purchase", { "send_to": "' . env('GTAG_SCRIPT_KEY') . '","transaction_id": "' . $invoice_no . '","affiliation": "' . $saleType . '","value": ' . $payment->price . ', "currency": "USD","items": [{"id": "' . $invoice_no . '", "name": "' . $payment->item_name . '","category": "' . $categoryNames . '","quantity": 1, "price": "' . $payment->price . '"},]});</script>';
            $gtag_script = $script . $logoDesign;


            return Inertia::render('Payment/SuccessPaypal', [
                "data" => Session()->get('success_paypal_data'),
                "payment" => $payment,
                'brand_settings' => $this->settingService->get(),
                'gtag_script' => $gtag_script
            ]);
        } else if (Session()->get('sendRequestFormData')) {

            $payment = PaymentLink::select('id', 'token')->where('token', Session()->get('sendRequestFormData')['formData']["token"])
                ->with([
                    'payments' => function ($payment) {
                        $payment->orderBy('id', 'ASC');
                        $payment->limit(1);
                        $payment->with('invoice');
                        $payment->with('customer');
                    }
                ]);

            if (env('IS_CATEGORY_ENABLED')) {
                $payment = $payment->with('categories');
            }

            $payment = $payment->first();

            $saleType = $payment->sale_type;
            $invoice_no = $payment->payments->invoice->invoice_no;
            $categories = $payment->categories;
            $categoryNames = $categories->pluck('name')->implode(' + ');

            $script = "<script async src='https://www.googletagmanager.com/gtag/js?id=" . env('GTAG_SCRIPT_KEY') . "'></script><script> window.dataLayer = window.dataLayer || []; function gtag(){dataLayer.push(arguments);} gtag('js', new Date()); gtag('config', '" . env('GTAG_SCRIPT_KEY') . "'); </script>";
            $logoDesign = '<script> gtag("event", "purchase", { "send_to": "' . env('GTAG_SCRIPT_KEY') . '","transaction_id": "' . $invoice_no . '","affiliation": "' . $saleType . '","value": ' . $payment->price . ', "currency": "USD","items": [{"id": "' . $invoice_no . '", "name": "' . $payment->item_name . '","category": "' . $categoryNames . '","quantity": 1, "price": "' . $payment->price . '"},]});</script>';
            $gtag_script = $script . $logoDesign;


            $formData = Session()->get('sendRequestFormData')['formData'];;
            $token = $request->token;
            $payerID = $request->PayerID;
            $accessToken = $this->generateAccessToken();

            $client = new Client();
            $response = $client->post("{$this->baseUrl}/v2/checkout/orders/{$token}/capture", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
            ]);

            //  dd($formData);
            // Save details in the User model (customer)
            $customer = User::firstOrCreate(
                ['email' => $formData['clientemail']],
                [
                    'first_name' => $formData['firstname'],
                    'last_name' => $formData['lastname'],
                    'phone' => $formData['phonenum'] ?? null,
                    'company' => $formData['companyname'] ?? null,
                    'address' => $formData['address'] ?? null,
                    'city' => $formData['city'] ?? null,
                    'state' => $formData['statename'] ?? null,
                    'zipcode' => $formData['zipcode'] ?? null,
                    'country' => $formData['country'] ?? null,
                    'created_by' => auth()->user()->id ?? null,
                    'created_at' => Carbon::now(),
                    'password' => Hash::make('12345678'), // Default password
                    'stripe_customer_id' => $paymentData['id'] ?? null, // Saving PayPal payment ID instead
                    'stripe' => 'paypal', // Indicating the payment method is PayPal
                    'stripe_pm_id' => $paymentData['payment_source']['paypal']['account_id'] ?? null, // Using PayPal account ID
                    'stripe_payment_method' => 'paypal', // Indicating PayPal as the payment method
                ]
            );

            // Assign customer role if it's a new user
            if ($customer->wasRecentlyCreated) {
                $customer->assignRole('Customer');

                //genertae log
                $log = array(
                    'activity' => 'Customer was Created',
                    'type' => 'customer.created',
                    'request' => response()->json($customer)->content(),
                    'response' => 201,
                    'code' => response()->json($customer)->status(),
                    'loggable_id' => $customer['id'],
                    'created_by' => auth()->id(),
                );
                $this->logRepository->log('customer', $log);
            }

            if (!$formData['itemid']) {
                return response()->json(['error' => 'Invalid payment link id.'], 400);
            }

            // save data in payment tbl
            // dd($paymentData);

            //  dd($formData);
            $paymentData = json_decode($response->getBody(), true);
            // Save payment details in the Payment model

            $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
            $location = $this->geoLocation->ipLocation($request->ip());
            $geoLocate = $this->neutrinoApi->geolocate(address: $formData['address'], houseNumber: "", street: "", city: $formData['city'], county: $formData['address'], state: $formData['statename'], postalCode: $formData['zipcode'], countryCode: "", languageCode: 'en', fuzzySearch: false);
            $geoAddress['geolocation'] = !empty($geoLocate['locations']) ? $geoLocate['locations'] : '';
            $locationResponse = !empty($location->body()) ? array_merge(json_decode($location->body(), true), $geoAddress) : $geoAddress;

            $updatePayment = [
                'location' => response()->json($locationResponse)->content(),
                'device' => response()->json($device)->content()
            ];

            $paymentcomplete = Payment::updateOrCreate([
                'customer_id' => $customer->id,
                'payment_link_id' => $formData['itemid'],
            ], [
                'price' => $paymentData['purchase_units'][0]['payments']['captures'][0]['amount']['value'] ?? 0.00,
                'discount' => $formData['discount'] ?? 0,
                'currency' => '170',
                'ip' => $request->ip(),
                'intent_id' => $paymentData['id'] ?? null,
                'charge_id' => $paymentData['purchase_units'][0]['payments']['captures'][0]['id'] ?? null,
                'charge' => json_encode($paymentData['purchase_units'][0]['payments']['captures'][0] ?? []),
                'refund' => null,
                'intent' => json_encode($paymentData ?? []),
                'paymentMethod' => json_encode($paymentData ?? []),
                'last_four' => 0000, // Not applicable for PayPal payments
                'bin' => null, // Not applicable for PayPal payments
                'coupon_id' => $formData['coupon_id'] ?? null,
                'converted_amount' => $paymentData['purchase_units'][0]['payments']['captures'][0]['amount']['value'] ?? 0.00,
                'comment' => null,
                'gateway_message' => $paymentData['status'] ?? null,
                'device' => $updatePayment['device'] ?? null,
                'location' => $updatePayment['location'] ?? null,
                'status' => ($paymentData['status'] === 'COMPLETED') ? '1' : '2',
                'created_at' => Carbon::now(),
            ]);

            $payment_id = $paymentcomplete->id;
            if ($paymentcomplete) {
                //generate log
                $paycomLog = array(
                    'activity' => 'Payment is Captured',
                    'type' => 'success',
                    'request' => response()->json([]),
                    'response' => response()->json($paymentcomplete)->content(),
                    'code' => response()->json($paymentcomplete)->status(),
                    'loggable_id' => $payment_id,
                    'created_by' => auth()->id(),
                );

                $this->logRepository->log('payment', $paycomLog);
            }

            InvoiceController::create($payment_id, false, 1);

            // Creating encrypted file for retry customer
            $paymentService->generateFile($formData['token'], $formData, $payment_id, auth()->id());

            // if ($paymentcomplete) {
            //     // Update the PaymentLink status to 2 after successful payment
            //     PaymentLink::where('id', $request->payment_link_id)->update(['status' => 2]);
            // }

            return Inertia::render('Payment/SuccessPaypal', [
                "data" => Session()->get('sendRequestFormData'),
                "payment_id" => $payment_id,
                "payment" => $payment,
                'brand_settings' => $this->settingService->get(),
                'gtag_script' => $gtag_script
            ]);
        } else {
            return redirect()->route('payment.expired');
        }
    }

    public function failed(Request $request, $data = "")
    {
        if (Session()->get('failed_data')) {
            $failedData = Session()->get('failed_data');
            $payment = PaymentLink::select('id', 'token')->where('token', $failedData['token'])->first();

            return Inertia::render('Payment/Error', [
                'brand_settings' => $this->settingService->get(),
                "data" => $failedData,
                "payment" => $payment
            ]);
        } else {
            return redirect()->route('payment.expired');
        }
    }

    public function queueApiLog($apiName, $request, $response, $paymentID, $createdBy, $delay)
    {
        dispatch(new ApiLogJob("A request to access ($apiName) api is initiated", 'api.call', response()->json($request)->content(), response()->json([])->content(), 200, $paymentID, $createdBy, $event = 1))->delay($delay);
        dispatch(new ApiLogJob('waiting_for_response', 'api.call.status', response()->json([])->content(), response()->json([])->content(), 200, $paymentID, $createdBy, $event = 1))->delay($delay);
        dispatch(new ApiLogJob("A request to retrieve data from ($apiName) API is completed", 'api.call', response()->json($request)->content(), $response, 200, $paymentID, $createdBy, $event = 0))->delay($delay);
    }
}
