<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use App\Classes\SendMail;
use App\Classes\SendTestInvoice;
use App\Helper\Helper\StripeHelper;
use App\Models\Category;
use App\Models\CountryCurrencies;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\PaymentLog;
use App\Models\Payments;
use App\Models\User;
use App\Services\LogService;
use App\Traits\ApplicationTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class InvoiceController extends Controller
{
  use ApplicationTrait;

  public static function create($payment_id = 1, $save_draft = false, $email_send = 0)
  {

    $payment_details = Payment::where('id', $payment_id)->with([
      'link' => function($link) {
        $link->select('id','created_by');
      },
      'customer' => function($customer) {
        $customer->select('id','first_name','last_name','email');
      },
      'currencyCountry' => function($currency) {
        $currency->select('id','aplha_code2','aplha_code3','country','currency','code','symbol');
      },
    ])->first();

    $now = Carbon::now();
    $invoice_date = $now->year . $now->month . $now->day;

    $last_inv = Invoice::latest('id')->first();

    if (!empty($last_inv)) {
      $invoice_no_str = explode("-", $last_inv->invoice_no);
      $last_id = $invoice_no_str[2];
      $invoice_no = "INV-" . $invoice_date . "-" . ($last_id + 1);
    } else {
      $invoice_no = "INV-" . $invoice_date . "-" . ($last_inv + 1 + 1000);
    }

    $invoice = new Invoice();
    $invoice->custom_id = StripeHelper::generateUniqueID('in', 10);
    $invoice->invoice_no = $invoice_no;
    $invoice->customer_id = (!empty($payment_details->customer_id)) ? ($payment_details->customer_id) : (1);
    $invoice->payment_id = $payment_id;
    $invoice->email_sent = $email_send;
    $invoice->created_by = $payment_details->link->created_by;
    $invoice->status = (!empty($save_draft)) ? (2) : (1);

    if($invoice->save()) {
      $invoice_id = $invoice->id;
      $invoice_details =  Invoice::find($invoice_id);
      $activity_log = $save_draft ? 'A draft invoice was created' : $payment_details->customer->email."'s payment for an invoice for " . $payment_details->currencyCountry->symbol . $payment_details->price . ' ' . $payment_details->currencyCountry->code . ' succeeded';
      $log = new PaymentLog();
      $log->activity = $activity_log;
      $log->type = 'created';
      $log->request = response()->json($invoice)->content();
      $log->response = response()->json($invoice)->content();
      $log->code = 200;
      $log->created_by = $invoice_details->created_by;
      $log->loggable()->associate($invoice_details);
      $log->save();
      if (!empty($invoice_details->email_sent)) {
        $emails = [
          'to' => $payment_details->customer->email,
          'cc' => ''
        ];
        
        if(app()->environment('production')) {
          (new InvoiceController)->mailTestInvoice($invoice_details->payment_id, $emails, app(LogService::class));
        }
      }
    }
  }

  public function mailTestInvoice($paymentID, $emails, LogService $logService) {
    $payment_info = Payment::select('id','customer_id','payment_link_id','price','discount','currency','ip','comment','status','created_at')
        ->where('id', $paymentID)
        ->with([
            'currencyCountry:id,code,symbol',
            'link' => function($link) {
                $link->select('id','token','customer_id','item_name','price','discount','discount_type','item_description','currency','status','created_at', 'created_by');
                $link->with('categories:id,name');
            }
        ])->first()->toArray();

    $brandSettings['settings'] = $this->get_setting()->toArray();
    $user_invoice = Invoice::select('id','invoice_no','custom_id','customer_id','payment_id','email_sent','created_by','status','created_at')
                    ->where('payment_id', $paymentID)->latest('id')->first()->toArray();
    $customer_info = User::select('id','first_name','last_name','email','phone','city','address','state','zipcode','country','company')
                    ->where('id', $payment_info['customer_id'])->first()->toArray();
    $category_info = $payment_info['link']['categories'];
    $currency_info = $payment_info['currency_country'];
    $createdAt = Carbon::parse($user_invoice['created_at']);
    $invoice_date = array('invoice_date' => $createdAt->format('d M Y') );
    $invoice_info = array_merge($brandSettings, $user_invoice, $customer_info, $payment_info, $payment_info['link'], $category_info, $invoice_date, $currency_info);

    $customer = User::find($customer_info['id']);

    if (!empty($invoice_info)) {
        if (!empty($customer_info)) {
            $sendMail = new SendTestInvoice($invoice_info, $customer, "Billing", ['emails'=> $emails['to'], 'ccEmails'=> $emails['cc']]);
            $sendMail->generate();
            $sendMail->mail();

            /*if (empty($user_invoice["email_sent"])) {
                return response()->json(['status' => false, "message" => "Invoice not sent."]);
            }*/

            $logService->log('invoice', [
                'activity' => $customer_info['email']."'s invoice for ". $payment_info['currency_country']['symbol'] . $payment_info['price'].' '. $payment_info['currency_country']['code'].' was sent to '.$emails['to'],
                'loggable_id' => $user_invoice['id'],
                'type' => 'invoice_send',
                'request' => response()->json($invoice_info)->content(),
                'response' => response()->json($invoice_info)->content(),
                'created_by' => $payment_info['link']['created_by'],
            ]);

            return ['success' => true, "message" => "Invoice sent."];
        } else {
            return ['success' => false, "message" => "Customer not exist."];
        }
    } else {
        return ['success' => false, "message" => "No Data"];
    }
  }

  public static function update($id, $payment_id = 1, $save_draft = false, $email_send = 0)
  {

    $payment_details = Payment::where('id', $payment_id)->with([
      'link' => function($link) {
        $link->select('id','created_by');
      },
      'customer' => function($customer) {
        $customer->select('id','first_name','last_name','email');
      },
      'currencyCountry' => function($currency) {
        $currency->select('id','aplha_code2','aplha_code3','country','currency','code','symbol');
      },
    ])->first();

    $invoice = Invoice::findOrFail($id);
    $invoice->customer_id = (!empty($payment_details->customer_id)) ? ($payment_details->customer_id) : (1);
    $invoice->payment_id = $payment_id;
    $invoice->email_sent = $email_send;
    $invoice->created_by = $payment_details->link->created_by;
    $invoice->status = (!empty($save_draft)) ? (2) : (1);

    if($invoice->update()) {
      $invoice_id = $invoice->id;
      $invoice_details =  Invoice::find($invoice_id);
      $activity_log = $save_draft ? 'A draft invoice was edited' : $payment_details->customer->email."'s invoice has changed";

      $log = new PaymentLog();
      $log->activity = $activity_log;
      $log->type = 'updated';
      $log->request = response()->json($invoice)->content();
      $log->response = response()->json($invoice)->content();
      $log->code = 200;
      $log->created_by = $invoice_details->created_by;
      $log->loggable()->associate($invoice_details);
      $log->save();

      if (!empty($invoice_details->email_sent)) {
        (new InvoiceController)->email_sent($invoice_details->customer_id);
      }
    }
  }

  public static function email_sent($user_id)
  {
    $customer = User::find($user_id);
    $brandSettings = (new InvoiceController)->get_setting()->toArray();
    $user_invoice = Invoice::where('customer_id', $user_id)->latest('id')->first()->toArray();
    $customer_info = User::find($user_invoice['customer_id'])->toArray();
    $payment_info = Payments::find($user_invoice['payment_id'])->toArray();
    $category_info = Category::find($payment_info['category_id'])->toArray();
    $currency_info = CountryCurrencies::find($payment_info['currency'])->toArray();
    $createdAt = Carbon::parse($user_invoice['created_at']);
    $invoice_date = array('invoice_date' => $createdAt->format('d M Y'));
    $invoice_info = array_merge($brandSettings, $user_invoice, $customer_info, $payment_info, $category_info, $invoice_date, $currency_info);

    if (!empty($invoice_info)) {

      if ($customer) {
        $sendMail = new SendMail($invoice_info, $customer, "Billing");
        $sendMail->generate();
        $sendMail->mail();

        if (empty($user_invoice["email_sent"])) {
          $invoice = Invoice::find($user_invoice['id']);
          $invoice->email_sent = 1;
          $invoice->save();

          $path = 'data/' . $payment_info["token"] . '.json';
          $file_content = file_get_contents($path);
          $content = json_decode($file_content, true);
          $formData["email_status"] = 1;
          $formData["email_error"] = "";

          $email_status = array_merge($content, $formData);
          $jsonString = json_encode($email_status, JSON_PRETTY_PRINT);
          file_put_contents($path, $jsonString);

          Log::info("Updated - Token: " . $payment_info["token"] . " has email_status: 1");
        }

        return ['status' => true, "message" => "Invoice sent."];
      } else {
        return ['status' => false, "message" => "Invoice not sent."];
      }
    } else {
      return ['status' => false, "message" => "No Data"];
    }
  }

  public static function payment_email_send($payment_id)
  {

    $brandSettings = (new InvoiceController)->get_setting()->toArray();
    $user_invoice = Invoice::where('payment_id', $payment_id)->first()->toArray();
    $customer_info = User::find($user_invoice['customer_id'])->toArray();
    $payment_info = Payments::find($user_invoice['payment_id'])->toArray();
    $category_info = Category::find($payment_info['category_id'])->toArray();
    $currency_info = CountryCurrencies::find($payment_info['currency'])->toArray();
    $createdAt = Carbon::parse($user_invoice['created_at']);
    $invoice_date = array('invoice_date' => $createdAt->format('d M Y'));
    $customer = User::find($user_invoice['customer_id']);
    $invoice_info = array_merge($brandSettings, $user_invoice, $customer_info, $payment_info, $category_info, $invoice_date, $currency_info);

    if (!empty($invoice_info)) {

      if ($customer) {
        $sendMail = new SendMail($invoice_info, $customer, "Billing");
        $sendMail->generate();
        $sendMail->mail();

        if (empty($user_invoice["email_sent"])) {
          $invoice = Invoice::find($user_invoice['id']);
          $invoice->email_sent = 1;
          $invoice->save();

          $path = 'data/' . $payment_info["token"] . '.json';
          $file_content = file_get_contents($path);
          $content = json_decode($file_content, true);
          $formData["email_status"] = 1;
          $formData["email_error"] = "";

          $email_status = array_merge($content, $formData);
          $jsonString = json_encode($email_status, JSON_PRETTY_PRINT);
          file_put_contents($path, $jsonString);

          Log::info("Updated - Token: " . $payment_info["token"] . " has email_status: 1");
        }

        return true;
      } else {
        return response()->json(['status' => false, "message" => "Invoice not sent."]);
      }
    } else {
      return response()->json(['status' => false, "message" => "No Data"]);
    }
  }
}
