<?php

namespace App\Http\Controllers\Admin;

use App\Classes\SmtpConfig;
use App\Helper\Helper\NotificationHelper;
use App\Helper\Helper\StripeHelper;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\CountryCurrencies;
use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Models\Team;
use App\Models\User;
use App\Models\LoginHistory;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Jenssegers\Agent\Agent;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use RingCentral\SDK\SDK;
use App\Models\Setting;
use Illuminate\Support\Facades\Mail;
use App\Mail\PasswordMail;
use App\Models\UserTeam;
use Illuminate\Support\Arr;
use App\Services\BinApiService;
use App\Services\LoginHistoryService;
use App\Services\Export\StaffExport;
use App\Services\Export\PaymentsExport;
use App\Helper\ActivityLogHelper;
use Illuminate\Support\Facades\Log;

class StaffController extends Controller
{
    protected $userImagePath = 'backend/images/upload/usersimg/';
    private $staffExport, $neutrinoApi, $loginHistoryService;
    protected $paymentsExport;

    function __construct(StaffExport $staffExport, BinApiService $neutrinoApi, LoginHistoryService $loginHistoryService, PaymentsExport $paymentsExport) {
        $this->staffExport = $staffExport;
        $this->neutrinoApi = $neutrinoApi;
        $this->loginHistoryService = $loginHistoryService;
        $this->paymentsExport = $paymentsExport;
    }

    public function list(Request $request)
    {
        $page['title'] = "Staff";

        //Begin::Columns Title
        $columns =  [
            [
                'key' => 'name',
                'component' => 'NameColumn',
                'text' => 'Name',
                'visible' => true,
                'fixed' => true,
            ],
            [
                'key' => 'email',
                'component' => 'EmailColumn',
                'text' => 'Email',
                'visible' => true,
            ],
            [
                'key' => 'phone',
                'component' => 'PhoneColumn',
                'text' => 'Phone',
                'visible' => true,
            ],
            [
                'key' => 'team',
                'component' => 'TeamColumn',
                'text' => 'Team',
                'visible' => true,
            ],
            [
                'key' => 'designation',
                'component' => 'DesignationColumn',
                'text' => 'Designation',
                'visible' => true,
            ],
            [
                'key' => 'status_counts',
                'component' => 'TotalPaymentColumn',
                'text' => 'No. of Payments',
                'visible' => true,
            ],
            [
                'key' => 'role',
                'component' => 'RoleColumn',
                'text' => 'Role',
                'visible' => true,
            ],
            [
                'key' => 'created_at',
                'component' => 'CreatedAtColumn',
                'text' => 'Created',
                'visible' => true,
            ],
        ];
        //End::Columns Title
        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;
        $statuses = StripeHelper::paymentStatuses();

        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Super Admin', 'Customer'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        $roles = $filteredRoles;

        // $getroles = Role::pluck('name')->toArray();

        // // Define roles to exclude
        // $excludedRoles = ['Super Admin', 'Customer'];

        // // Filter out roles excluding the excluded roles
        // $filteredRoles = array_diff($getroles, $excludedRoles);

        // // If you need to re-index the array after filtering
        // $filteredRoles = array_values($filteredRoles);

        // $roles = $filteredRoles;

        // if ($user->can('User-Global') || $authUserRole === 'Super Admin') {

        //     $rolesWithPaymentGlobalPermission = Role::whereHas('permissions', function ($query) {
        //         $query->where('name', 'Payment-Global');
        //     })->pluck('name')->toArray();

        //     // If the authenticated user is an admin, include the 'Admin' role in the roles array
        //     $roles = $rolesWithPaymentGlobalPermission;
        // }

        // if ($authUserRole === 'Super Admin') {
        //     // If the authenticated user is an admin, include the 'Super Admin' role in the roles array
        //     $roles[] = 'Super Admin';
        // }
        //Begin::Getting Staff from users table
        $staffs = User::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'designation', 'created_by', 'created_at')

            ->with('teams:id,name,target,color')
            ->with('roles:id,name')
            ->when(Auth::user()->roles->pluck('name')[0] !== 'Super Admin', function ($query) {
                $query->whereHas('roles', function ($q) {
                    $q->whereNotIn('name', ['Customer', 'Super Admin']);
                });
            }, function ($query) {
                $query->whereHas('roles', function ($q) {
                    $q->where('name', '!=', 'Customer');
                });
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('first_name', 'like', '%' . $request->name . '%')
                    ->orWhere('last_name', 'like', '%' . $request->name . '%');
            })
            ->when(!empty($request->email), function ($query) use ($request) {
                $query->where('email', 'like', '%' . $request->email . '%');
            })
            ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
            })
            ->when(!empty($request->designation), function ($query) use ($request) {
                $query->where('designation', $request->designation);
            })
            ->when(!empty($request->team), function ($query) use ($request) {
                $query->whereHas('teams', function ($q) use ($request) {
                    $q->where('name', $request->team);
                });
            })
            ->when(!empty($request->role), function ($query) use ($request) {
                $query->whereHas('roles', function ($q) use ($request) {
                    $q->where('name', $request->role);
                });
            })
            ->when(!empty(Auth::user()->roles->pluck('name')[0]) && (Auth::user()->roles->pluck('name')[0] !== 'Super Admin' && !$authUser->can('User-Global')), function ($query) use ($authUser) {
                $query->where('id', $authUser->id);
            })
            ->paginate();

            $staffIds = $staffs->pluck('id');

            // Initialize an array to hold the status counts
            $staffStatusCounts = [];

            // Fetch status counts for each staff
            foreach ($staffIds as $staffId) {
                $statusCounts = Payment::userStatusCount($statuses, User::find($staffId));
                $staffStatusCounts[$staffId] = $statusCounts;
            }

            // Add status counts to each staff member
            $staffs->getCollection()->transform(function ($staff) use ($staffStatusCounts) {
                $staff->status_counts = $staffStatusCounts[$staff->id] ?? [];
                return $staff;
            });

        //End::Getting Staff from users table

        //Begin::Getting Teams from teams table
        $teams = Team::get();
        //End::Getting Teams from teams table

        //Begin::Getting Roles from teams table

        $roles = Role::where('name','!=', 'Customer')->get();

        //End::Getting Roles from teams table

        $countries = CountryCurrencies::select('id', 'aplha_code2', 'currency', 'code', 'symbol', 'country')
            ->orderBy('currency', 'ASC')
            ->get();

        $exportColumns = $this->staffExport->columns();

        //Begin::Inertia Response
        return Inertia::render('Admin/Staff/List', [
            'page' => $page,
            'columns' => $columns,
            'staff' => $staffs,
            'teams' => $teams,
            'roles' => $roles,
            'countries' => $countries,
            'authUserRole' => $authUserRole,
            'exportColumns' => $exportColumns
        ]);
        //End::Inertia Response
    }

    public function detailOverview(Request $request)
    {
        $userId = $request->id ? $request->id : $request->user()->id;

        //Begin::Getting Current USer from Users Table by its ID
        $staff = User::select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'address', 'company', 'country', 'state', 'city', 'zipcode', 'designation', 'facebook', 'instagram', 'linkedin', 'two_step', 'two_step_created', 'created_by', 'created_at')
            ->with('teams')
            ->find($userId);
        //End::Getting Current USer from Users Table by its ID

        $userRole = $staff?->roles?->first()?->name;
        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        $currentDate['date'] = Carbon::now()->format('Y-m-d');

        //Begin::Getting Dates for the current month
        $dateFrom = Carbon::now()->startOfMonth()->format('Y-m-d');
        $dateTo = Carbon::now()->endOfMonth()->format('Y-m-d');
        //End::Getting Dates for the current month

        //Begin::Payment Graph Begin Data
        $graphPayments = [
            'payments' => 0,
            'failedpayments' => 0,
            'upsell' => 0,
            'newsale' => 0,
            'incomplete' => 0,
            'customers' => 0,
            'spendpercustomer' => 0,
        ];
        //End::Payment Graph Begin Data

        //Begin::Getting Payment Data
        $allPayments = Payment::select('id', 'payment_link_id', 'price', 'status')->whereHas('link', function ($link) use ($request, $userId) {
            $link->where('created_by', $userId);
        })->whereDate('created_at', '>=', $dateFrom)->whereDate('created_at', '<=', $dateTo)->get();

        $spendpercustomer = Payment::select('id', 'payment_link_id', 'price', 'status')->whereHas('link', function ($link) use ($request, $userId) {
            $link->where('created_by', $userId);
            $link->where('status', 2);
        })->where('status', 1)->whereDate('created_at', '>=', $dateFrom)->whereDate('created_at', '<=', $dateTo)->get();
        //End::Getting Payment Data

        $freshsalesCount = $this->getSaleTypeCountFilterById("fresh_sales", $dateFrom, $dateTo, $userId);
        $upsellCount = $this->getSaleTypeCountFilterById("upsell", $dateFrom, $dateTo, $userId);

        $graphPayments['newsale'] = $freshsalesCount['count'] ?? 0;
        $graphPayments['upsell'] = $upsellCount['count'] ?? 0;

        //Begin::Setting Data for the graph
        $graphPayments['payments'] = number_format($allPayments->where('status', 1)->sum('price'));
        $graphPayments['failedpayments'] = number_format($allPayments->where('status', 2)->sum('price'));
        $graphPayments['incomplete'] = number_format($allPayments->where('status', 3)->sum('price'));
        $graphPayments['customers'] = number_format($allPayments->count('price'));
        $graphPayments['spendpercustomer'] = number_format($spendpercustomer->count() > 0 ? $spendpercustomer->sum('price') / $spendpercustomer->count() : 0);
        //End::Setting Data for the graph

        //Begin::Getting Total Balance of current user
        $payments = Payment::select('id', 'payment_link_id', 'price')->whereHas('link', function ($link) use ($request, $userId) {
            $link->where('created_by', $userId);
        })->where('status', 1);
        $balanceAmount = number_format($payments->sum('price'));
        $monthlyBalanceAmount = number_format($payments->whereMonth('created_at', Carbon::today())->sum('price'));
        $dailyBalanceAmount = number_format($payments->whereDate('created_at', Carbon::today())->sum('price'));
        //End::Getting Total Balance of current user

        //Begin::Saving User Balance in an array
        $userBalance = [
            'totalBalance' => $balanceAmount,
            'monthBalance' => $monthlyBalanceAmount,
            'dayBalance' => $dailyBalanceAmount
        ];


        //End::Saving User Balance in an array
        $userSession = [];
        $sessionData = LoginHistory::query()
                    ->where('user_id', $userId)
                    ->latest()
                    ->take(4)
                    ->get();
// dd($sessionData);

        $sessionData->map(function ($session) use (&$userSession) {
            // $agent = new Agent();
            // $agent->setUserAgent($session->user_agent);
            // $data = unserialize(base64_decode($session->payload));

            // $ipInfo = $this->neutrinoApi->ipInfo($session->ip_address);
            // $country = $ipInfo['country'] ?? '';

            $currentTime = now();
            $createdTime = Carbon::parse($session->created_at);

            $durationInSeconds = $currentTime->diffInSeconds($createdTime);
            $hours = floor($durationInSeconds / 3600);
            $minutes = floor(($durationInSeconds % 3600) / 60);
            $seconds = $durationInSeconds % 60;

            if ($hours > 0) {
                $formattedDuration = $hours . ($hours == 1 ? ' hour' : ' hours');
                if ($minutes > 0) {
                    $formattedDuration .= ', ' . $minutes . ' minutes';
                }
            } elseif ($minutes > 0) {
                $formattedDuration = $minutes . ' minutes';
            } else {
                $formattedDuration = $seconds . ' seconds';
            }

            $userSession[] = [
                'id' => $session->session_id,
                'user_id' => $session->user_id,
                'ip_address' => $session->ip_address,
                'country' => $session->country,
                // 'user_agent' => $session->user_agent,
                'device' => $session->device,
                'deviceType' => $session->device_type,
                'platform' => $session->platform,
                // 'isMobile' => $agent->isPhone(),
                // 'isDesktop' => $agent->isDesktop(),
                'browser' => $session->browser,
                'duration' => $session->duration,
                // 'last_activity' => $session->last_activity,
                'session_created_at' => !empty($session->duration) ? $session->duration : $formattedDuration,
                'current_session' => Session::getId() == $session->session_id
            ];
        });

        $countries = CountryCurrencies::select('id', 'aplha_code3', 'currency', 'code', 'symbol', 'country')
            ->orderBy('currency', 'ASC')
            ->get();

        $teams = Team::select('id', 'name')->get();

        $roles = Role::where('name','!=', 'Customer')->get();

        return Inertia::render('Admin/Staff/DetailOverView', [
            'staff' => $staff,
            'userRole' => $userRole,
            'userBalance' => $userBalance,
            'graphPayments' => $graphPayments,
            'sessionData' => $userSession,
            'countries' => $countries,
            'teams' => $teams,
            'roles' => $roles,
            'authUserRole' => $authUserRole,
            'currentDate' => Carbon::createFromDate($currentDate['date'])->format('d/m/Y'),
        ]);
    }

    public function detailPermission(Request $request)
    {
        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Customer'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        //Begin::Getting Current User Role
        $user = User::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'created_by', 'created_at')
            ->role($filteredRoles)
            ->with('teams:id,name,target,color')
            ->with('roles:id,name')
            ->find($request->id);

        $roles = $user->roles->pluck('id')->toArray();
        //End::Getting Current User Role
        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;
        //Begin::Allowed permissions id as role
        $permissionRole = DB::table("role_has_permissions")
            ->whereIn("role_has_permissions.role_id", $roles)
            ->pluck('role_has_permissions.permission_id')
            ->toArray();
        //End::Allowed permissions id as role

        //Begin::Getting Permission for frontEnd Display
        $permissions = Permission::select('id', 'name', 'guard_name')->get();
        //End::Getting Permission for frontEnd Display

        $permissionAllowed = [];

        //Begin::Getting First Key of an Array for grouping, taking allowed permission to current user in AllowedPermission Array
        foreach ($permissions as $permission) {
            // $currentPermission = explode("-", $permission->name);

            if (in_array($permission->id, $permissionRole)) {
                $permissionAllowed[] = $permission->id;
            }
        }
        //End::Getting First Key of an Array for grouping, taking allowed permission to current user in AllowedPermission Array

        $groupedPermissions = [];

        $groupKeyDisplayNames = [
            'TwoStep' => '2 Step Authentication',
            'ImportExport' => 'Export',
            'BriefLink' => 'Brief Link',
            'BriefForm' => 'Brief Form',
        ];

        //Begin::Grouping Permissions as Category with the first key of exploded permission, checking permission ID is in permission Allowed then add it as Alowe key
        foreach ($permissions as $permission) {
            $currentPermission = explode("-", $permission->name);
            $groupKey = $currentPermission[0];

            // Skip user and coupon permissions
            if ($groupKey === 'Coupon' || $groupKey == 'Settings' || $groupKey == 'Gateway') {
                continue;
            }

            $displayName = $groupKeyDisplayNames[$groupKey] ?? $groupKey;

            if (!isset($groupedPermissions[$groupKey])) {
                $groupedPermissions[$groupKey] = [
                    'groupDisplayName' => $displayName,
                    'permissions' => [
                        'global' => false,
                        'globalAllow' => false,
                        'create' => false,
                        'createAllow' => false,
                        'edit' => false,
                        'editAllow' => false,
                        'view' => false,
                        'viewAllow' => false,
                        'delete' => false,
                        'deleteAllow' => false,
                    ],
                ];
            }

            $action = strtolower($currentPermission[1]);

            $groupedPermissions[$groupKey]['permissions'][$action] = true;
            $groupedPermissions[$groupKey]['permissions'][$action . 'Allow'] = in_array($permission->id, $permissionAllowed);
            $groupedPermissions[$groupKey]['permissions'][$action . 'Id'] = $permission->id;
        }
        //Begin::Grouping Permissions as Category with the first key of exploded permission, checking permission ID is in permission Allowed then add it as Alowe key

        //Begin::Getting Current USer from Users Table by its ID
        $staff = User::select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'created_by', 'created_at')->find($request->id);
        //End::Getting Current USer from Users Table by its ID

        return Inertia::render('Admin/Staff/DetailPermission', [
            'staff' => $staff,
            'totalPermissions' => $groupedPermissions,
            'authUserRole' => $authUserRole,
        ]);
    }

    public function detailPayment(Request $request)
    {
        $page["title"] = "Payment List - PaymentModule";

        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Customer'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        //Begin::Getting Current User Role
        $user = User::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'created_by', 'created_at')
            ->role($filteredRoles)
            ->with('teams:id,name,target,color')
            ->with('roles:id,name')
            ->find($request->id);

        $roles = $user->roles->pluck('id')->toArray();
        //End::Getting Current User Role

        //Begin::Allowed permissions id as role
        $permissionRole = DB::table("role_has_permissions")
            ->whereIn("role_has_permissions.role_id", $roles)
            ->pluck('role_has_permissions.permission_id')
            ->toArray();
        //End::Allowed permissions id as role

        //Begin::Getting Permission for frontEnd Display
        $permissions = Permission::select('id', 'name', 'guard_name')->get();
        //End::Getting Permission for frontEnd Display

        $permissionAllowed = [];

        //Begin::Getting First Key of an Array for grouping, taking allowed permission to current user in AllowedPermission Array
        foreach ($permissions as $permission) {
            // $currentPermission = explode("-", $permission->name);

            if (in_array($permission->id, $permissionRole)) {
                $permissionAllowed[] = $permission->id;
            }
        }
        //End::Getting First Key of an Array for grouping, taking allowed permission to current user in AllowedPermission Array

        $groupedPermissions = [];

        //Begin::Grouping Permissions as Category with the first key of exploded permission, checking permission ID is in permission Allowed then add it as Alowe key
        foreach ($permissions as $permission) {
            $currentPermission = explode("-", $permission->name);
            $groupKey = $currentPermission[0];

            if (!isset($groupedPermissions[$groupKey])) {
                $groupedPermissions[$groupKey] = [
                    'permissions' => [
                        'create' => false,
                        'createAllow' => false,
                        'edit' => false,
                        'editAllow' => false,
                        'view' => false,
                        'viewAllow' => false,
                        'delete' => false,
                        'deleteAllow' => false,
                    ],
                ];
            }

            $action = strtolower($currentPermission[1]);

            $groupedPermissions[$groupKey]['permissions'][$action] = true;
            $groupedPermissions[$groupKey]['permissions'][$action . 'Allow'] = in_array($permission->id, $permissionAllowed);
            $groupedPermissions[$groupKey]['permissions'][$action . 'Id'] = $permission->id;
        }
        //Begin::Grouping Permissions as Category with the first key of exploded permission, checking permission ID is in permission Allowed then add it as Alowe key

        //Begin::Getting Current USer from Users Table by its ID
        $staff = User::select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'created_by', 'created_at')
            ->with('roles:id,name')
            ->find($request->id);
        //End::Getting Current USer from Users Table by its ID

        //Begin::Columns for Payment Table
        $columns = [
            ['key' => 'price', 'component' => 'AmountColumn', 'text' => 'Amount', 'visible' => true, 'fixed' => true, 'order' => 'desc'],
            ['key' => 'link.item_name', 'component' => 'DescriptionColumn', 'text' => 'Item Name', 'visible' => true, 'order' => 'desc'],
            ['key' => 'customer.email', 'component' => 'CustomerColumn', 'text' => 'Customer', 'visible' => true, 'order' => 'desc'],
            ['key' => 'gateway.name', 'component' => 'MerchantColumn', 'text' => 'Payment Gateway', 'visible' => false, 'order' => 'desc'],
            ['key' => 'card', 'component' => 'CardColumn', 'text' => 'Card', 'visible' => false, 'order' => 'desc'],
            ['key' => 'link.sale_type', 'component' => 'BadgeColumn', 'text' => 'Sales Type', 'visible' => true, 'color' => '#625afa', 'order' => 'desc'],
        ];
       
        if ($user->hasRole('Super Admin') || ($user->can('Payment-Global'))) {
            $columns[] = ['key' => 'link.creator.first_name', 'component' => 'BadgeColumn', 'text' => 'Salesperson', 'visible' => true, 'order' => 'desc'];
        }

        $columns[] = ['key' => 'link.team_wise.name', 'component' => 'BadgeColumn', 'text' => 'Team', 'colorKey' => 'link.team_wise.color', 'visible' => false, 'order' => 'desc'];
        $columns[] = ['key' => 'comment', 'component' => 'CommentColumn', 'text' => 'Comment', 'visible' => false, 'order' => 'desc'];
        $columns[] = ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true, 'order' => 'desc'];

        if ($request->order && $request->key) {
            $columns = array_map(function($column) use ($request) {
                if ($column['key'] === $request->key) {
                    $column['order'] = $request->order;
                }
                return $column;
            }, $columns);
        }
        //End::Colmns for Payment Table

        //Begin::Currencies for Payment table
        $currencies = CountryCurrencies::select('id', 'currency', 'code', 'symbol')->groupBy('code')->get();
        //End::Currencies for Payment table

        //Begin::Getting Statuses
        $statuses = StripeHelper::paymentStatuses();
        //End::Getting Statuses

        $teams = Team::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();

        //Begin::Payments Records of current User
        $baseQuery = Payment::baseQuery();
        $baseQuery = $baseQuery
            ->when(!empty($request->is_review), function ($query) use ($request) {
                $unReviewedPayments = Notification::selectRaw('json_extract(data, "$.data.id") as id')
                    ->whereNull('read_at')
                    ->where('data->type', 'payment')
                    ->where('notifiable_id', Auth::id())
                    ->get();
                $query->where('status', 1);
                $query->whereIn('id', $unReviewedPayments);

                $unReviewedIds = $unReviewedPayments->pluck('id')->toArray();
                $this->markAllPaymentAsRead($unReviewedIds);
            })
            ->when(!empty($request->status), function ($query) use ($request, $statuses) {
                $mergeStatus = array_intersect($request->status, $statuses);
                $currentStatus = array_map(function ($status) use ($statuses) {
                    return array_search($status, $statuses);
                }, $mergeStatus);
                if (!empty($currentStatus)) {
                    $query->whereIn('status', $currentStatus);
                }
            })
            ->filterByAmount($request, 'amountSearch', 'amountSearchBy')
            ->filterByCurrency($request)
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->filterByCustomerData($request)
            ->filterByPaymentLink($request)
            ->filterByLastFour($request)
            ->filterByCard($request)
            ->when(!empty($request->order) , function ($query) use ($request) {
                if(!empty($request->order) && !empty($request->key)){
                    $query->orderBy($request->key, $request->order);
                }else{
                    $query->orderBy('created_at', $request->order);
                }
            })
            ->whereHas('link', function ($link) use ($request) {
                $link->where('created_by', $request->id);
            });

        
        //End::Payments Records of current User

        $cloneBaseQuery = clone $baseQuery;

        $baseQuery = $baseQuery->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
            if (in_array($request->status_all, ['fresh_sales', 'upsell'])) {
                $query->whereHas('link', function ($q) use ($request) {
                        $q->where('sale_type', $request->status_all);
                });
            } else {
            
                $currentStatus = 0;
                foreach ($statuses as $key => $status) {
                    if ($status == $request->status_all) {
                        $currentStatus = $key;
                    }
                }
                if ($currentStatus > 0) {
                    $query->where('status', $currentStatus);
                }
            }
        });

        $payment = (clone $baseQuery)
            ->when(!empty($request->dateSearchBy), function ($query) {
                $query->orderBy('created_at', 'asc');
            }, function ($query) {
                $query->latest('id');
            })
            ->paginate(20)
            ->withQueryString();

        $otherInformation = Payment::getOtherInformation(clone $cloneBaseQuery, $statuses, $request);
         //Begin::Payment Similar Status Count in Grouping
        $paymentSums = Payment::queryStatusCount($cloneBaseQuery, $statuses, ($user->hasAnyRole($roles) ? $user->id : ""), $user->roles->pluck('name')[0], $request);
        //End::Payment Similar Status Count in Grouping

        $gateways = PaymentGateway::select('id', 'name', 'gateway', 'environment', 'created_by')->get();
        $exportColumns = $this->paymentsExport->columns();

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        return Inertia::render('Admin/Staff/DetailPayment', [
            'staff' => $staff,
            'teams' => $teams,
            'categories' => $categories,
            'payments' => $paymentSums->merge($payment),
            'columns' => $columns,
            'page' => $page,
            'currencies' => $currencies,
            'gateways' => $gateways,
            'authUserRole' => $authUserRole,
            'exportColumns' => $exportColumns,
            'otherInformation' => $otherInformation,
        ]);
    }

    public function permission(Request $request)
    {
        $data['success'] = true;
        try {
            $admin = User::query()
                ->select('id', 'first_name', 'last_name', 'email', 'phone', 'image', 'created_by', 'created_at')
                ->find($request->user()->id);
            if (!$admin->hasRole('Super Admin')) {
                $data['message'] = 'You are not an Super Admin';
                return response()->json($data);
            }

            $user = User::query()
                ->select('id', 'first_name', 'last_name', 'email')
                ->find($request->userId);

            $staffRole = $user->getRoleNames()->first();
            $role = Role::findByName($staffRole);
            $permission = Permission::findById($request->permissionId);

            $modulePermit = explode('-', $permission->name);
            if ($role->hasPermissionTo($permission->name)) {
                $role->revokePermissionTo($permission->name);
                $data['message'] = $user->first_name . '\'s access revoked on ' . $modulePermit[1] . ' of ' . $modulePermit[0] . ' module';
            } else {
                $role->givePermissionTo($permission->name);
                $data['message'] = $user->first_name . '\'s access granted on ' . $modulePermit[1] . ' of ' . $modulePermit[0] . ' module';
            }
            NotificationHelper::notify('permission', $admin->first_name . ' ' . 'has updated your permission' . ' ' . $permission->name, [
                'id' => $permission->id,
                'permission' =>  $permission,
                'message' => 'Permission successful',
            ], Auth::id());
        } catch (Exception $e) {
            $data['success'] = false;
            $data['message'] = $e->getMessage();
        }

        return response()->json($data);
    }

    public function signoutStaffSessions(Request $request)
    {
        $data['success'] = true;
        try {
            $authUser = Auth::user();
            $user = User::findOrNew($request->staffID);
            if ($user && $user->two_step == 1) {
                $user->google2fa_secret = null;
                $user->save();
            }

            $sessionData = DB::table('sessions')->select('id', 'user_id', 'ip_address', 'user_agent', 'last_activity')
                ->where('user_id', $request->staffID)
                ->get();

            foreach ($sessionData as $session) {
                $sessionHandler = Session::getHandler();
                $sessionHandler->destroy($session->id);
                $createLogoutHistory = $this->loginHistoryService->update($request->staffID, $session->id);

            }
            if ($authUser->id != $request->staffID) {
                $data['staff_id'] = $request->staffID;
            }

            $data['message'] = 'Signed out from all sessions';

            return response()->json($data);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function update(Request $request)
    {
        if (!empty($request->phone)) {
            $requestPhone = str_replace(' ', '', $request->phone);
            $request->merge(['phone' => $requestPhone]);
        }

        $rules = [
            'first_name' => 'nullable|string',
            'last_name' => 'nullable|string',
            'phone' => 'nullable|string',
            // 'company' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'zipcode' => 'nullable|string',
            'password' => 'nullable|string|min:8',
            'facebook' => 'nullable|string',
            'instagram' => 'nullable|string',
            'linkedin' => 'nullable|string',
            'designation' => 'nullable|string',
            'country' => 'nullable|string',
            'team' => 'nullable|integer',
        ];

        if ($request->hasFile('image')) {
            $rules['image'] = 'image|mimes:jpeg,png,jpg,gif|max:2048';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ]);
        } else {

            $user = User::find($request->id);

            // if ($request->filled('first_name')) {
            $user->first_name = $request->input('first_name');
            // }

            // if ($request->filled('last_name')) {
            $user->last_name = $request->input('last_name');
            // }

            // if ($request->filled('phone')) {
            $user->phone = $request->input('phone');
            // }

            // if ($request->filled('company')) {
            //     $user->company = $request->input('company');
            // }

            // if ($request->filled('address')) {
            $user->address = $request->input('address');
            // }

            // if ($request->filled('city')) {
            $user->city = $request->input('city');
            // }

            // if ($request->filled('state')) {
            $user->state = $request->input('state');
            // }

            // if ($request->filled('zipcode')) {
            $user->zipcode = $request->input('zipcode');
            // }

            // if ($request->filled('facebook')) {
            $user->facebook = $request->input('facebook');
            // }

            // if ($request->filled('instagram')) {
            $user->instagram = $request->input('instagram');
            // }

            // if ($request->filled('linkedin')) {
            $user->linkedin = $request->input('linkedin');
            // }

            // if ($request->filled('designation')) {
            $user->designation = $request->input('designation');
            // }

            // if ($request->filled('country')) {
            $user->country = $request->input('country');
            // }

            // if ($request->filled('company')) {
                $user->company = Setting::where('key', 'company_name')->get('key');
            // }

            if ($request->filled('password')) {
                $user->password =  Hash::make($request->input('password'));
                $user->password_expire = 0;

                $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');
                $data = array(
                    "email" => !empty($request->email) ? $request->email : '',
                    "brand_settings" => !empty($brand_settings) ? $brand_settings : '',
                    "password" => !empty($request->password) ? $request->password : '',

                );

                $setSMTP = new SmtpConfig('general', env('BILLING_EMAIL'));

                if (!empty($user->email)) {
                    Mail::to($user->email)->send(new PasswordMail($data));
                }
            }

            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '.' . $image->getClientOriginalExtension();
                $imagePath = $this->userImagePath . $imageName;
                $image->move($this->userImagePath, $imageName);
                $user->image = $imagePath;
            }

            $user->save();
            if ($request->team !== '0') {
                $userTeam = UserTeam::where('user_id', $request->id)->first();

                if ($userTeam) {
                    // If the user team record exists and team data is provided, update it
                    if ($request->filled('team')) {
                        $userTeam->team_id = $request->team;
                        $userTeam->created_by = Auth::id();
                        $userTeam->save();
                    }
                } else {
                    // If the user team record doesn't exist and team data is provided, create a new one
                    if ($request->filled('team')) {
                        $userTeam = new UserTeam();
                        $userTeam->user_id = $request->id;
                        $userTeam->team_id = $request->team;
                        $userTeam->created_by = Auth::id();
                        $userTeam->save();
                    }
                }
            }
            // else {
            //     $userTeam = UserTeam::where('user_id', $request->id)->first();

            //     if ($userTeam) {
            //         $userTeam->forceDelete();
            //     }
            // }

            $role = Auth::user()->getRoleNames()->first();
            NotificationHelper::notify('staff', $role . ' has updated' . ' ' . $user->first_name . ' ' . $user->last_name . ' ' . 'account information ', [
                'id' => $request->id,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($request->all())->content(),
                'message' => 'staff Updated Successful',
            ], Auth::id());

            $currentRole = $user?->roles?->first()?->name;
            //$roleUppercase = ucwords(str_replace("_", " ", $request->role));

            if ($currentRole != $request->role) {
                $currentRoles = $user->getRoleNames();

                foreach ($currentRoles as $role) {
                    $user->removeRole($role);
                }
            }

            if ($request->role != null) {
                $user->assignRole($request->role);
            }

            $authUser = Auth::user();

            if ($request->filled('password')) {
                $user = User::findOrNew($request->id);
                if ($user && $user->two_step == 1) {
                    $user->google2fa_secret = null;
                    $user->save();
                }

                $sessionData = DB::table('sessions')->select('id', 'user_id', 'ip_address', 'user_agent', 'last_activity')
                    ->where('user_id', $request->id)
                    ->get();

                foreach ($sessionData as $session) {
                    $sessionHandler = Session::getHandler();
                    $sessionHandler->destroy($session->id);

                    $createLogoutHistory = $this->loginHistoryService->update($request->id, $session->id);
                }
                if ($authUser->id == $request->id) {
                    session()->flush();
                    return Redirect::route('admin.login');
                }
            }

            if ($request->returnJson) {
                return response()->json(['success' => true, 'message' => 'Staff Updated Successful']);
            } else {
                return Redirect::route('admin.staff.detail.overview', ['id' => $user->id]);
            }
        }
    }

    public function create(Request $request)
    {
        if (!empty($request->phone)) {
            $requestPhone = str_replace(' ', '', $request->phone);
            $request->merge(['phone' => $requestPhone]);
        }

        $rules = [
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|unique:users,phone',
            // 'company' => 'required|string',
            'role' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'zipcode' => 'nullable|string',
            'password' => 'nullable|string|min:8',
            'facebook' => 'nullable|string',
            'instagram' => 'nullable|string',
            'linkedin' => 'nullable|string',
            'designation' => 'nullable|string',
            'country' => 'nullable|string',
            'team' => 'nullable|integer',
        ];

        if ($request->hasFile('image')) {
            $rules['image'] = 'image|mimes:jpeg,png,jpg,gif|max:2048';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ]);
        } else {

            $user = new User();

            $user->first_name = $request->first_name;
            $user->last_name = $request->last_name;
            $user->phone = $request->phone;
            $user->email = $request->email;
            // $user->company = $request->company;
            $user->address = $request->address;
            $user->city = $request->city;
            $user->state = $request->state;
            $user->zipcode = $request->zipcode;
            $user->facebook = $request->facebook;
            $user->instagram = $request->instagram;
            $user->linkedin = $request->linkedin;
            $user->designation = $request->designation;
            $user->country = $request->country;
            $user->company = Setting::where('key', 'company_name')->get('key');
            $user->password = Hash::make($request->password);
            $user->password_expire = 1;

            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '.' . $image->getClientOriginalExtension();
                $imagePath = $this->userImagePath . $imageName;
                $image->move($this->userImagePath, $imageName);
                $user->image = $imagePath;
            }

            $user->save();
            if ($request->team !== '0') {
                $userTeam = new UserTeam();

                $userTeam->user_id = $user->id;
                $userTeam->team_id = $request->team;
                $userTeam->created_by = Auth::id();
                $userTeam->save();
            }

            $roll = Auth::user()->getRoleNames()->first();
            NotificationHelper::notify('staff', $roll . ' has created' . ' ' . $user->first_name . ' ' . $user->last_name . ' new staff account ', [
                'id' => $request->id,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($request->all())->content(),
                'message' => 'staff Created Successful',
            ], Auth::id());

            if ($request->role != null) {
                $user->assignRole($request->role);
            }

            $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');
            $data = array(
                "email" => !empty($request->email) ? $request->email : '',
                "brand_settings" => !empty($brand_settings) ? $brand_settings : '',
                "password" => !empty($request->password) ? $request->password : '',

            );

            $setSMTP = new SmtpConfig('general', env('BILLING_EMAIL'));

            if (!empty($request->email)) {
                Mail::to($request->email)->send(new PasswordMail($data));
                return true;
            }

            if ($request->returnJson) {
                return response()->json(['success' => true, 'message' => 'Staff Created Successful']);
            } else {
                return Redirect::route('admin.staff.list', ['id' => $user->id]);
            }
        }
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:users,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:users,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        // Define roles to include
        $includedRoles = ['Super Admin'];

        // Retrieve roles that have the 'Payment-Global' permission
        $rolesWithPaymentGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'PaymentLinkGenerator-Global');
        })->pluck('name')->toArray();

        // Combine included roles and roles with 'Payment-Global' permission
        $filteredRoles = array_merge($includedRoles, $rolesWithPaymentGlobalPermission);

        $roles = $filteredRoles;


        if ($request->user()->hasAnyRole($roles)) {

            try {
                if ($request->filled('id')) {
                    $userTeam = UserTeam::where('user_id', $request->id)->first();
                    $userTeam->forceDelete();
                    $staff = User::findOrFail($request->id);
                    $staff->forceDelete();
                } else if ($request->filled('ids')) {
                    $ids = $request->ids; // Array with one element
                    $userTeam = UserTeam::whereIn('user_id', $ids)->forceDelete();
                    User::withTrashed()->whereIn('id', $ids)->forceDelete();
                }

                return response()->json([
                    'success' => true,
                    'message' => "Staff(s) successfully deleted.",
                ], 200);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => "An error occurred while deleting staff(s).",
                ], 500);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => "You are not allowed to perform this operation.",
            ], 403);
        }
    }

    public function sendOtp(Request $request)
    {
        if (!empty($request->phoneNumber)) {
            $requestPhone = str_replace(' ', '', $request->phoneNumber);
            $request->merge(['phoneNumber' => $requestPhone]);
        }

        $RECIPIENT =  $request->phoneNumber;
        // Generate a 6-digit random OTP
        $otp = mt_rand(100000, 999999);

        // Instantiate the SDK and get the platform instance
        $rcsdk = new SDK(getenv('RINGCENTRAL_CLIENT_ID'), getenv('RINGCENTRAL_CLIENT_SECRET'), getenv('RINGCENTRAL_SERVER_URL'));
        $platform = $rcsdk->platform();

        // Authenticate a user using a personal JWT token
        try {
            $platform->login(['jwt' => 'eyJraWQiOiI4NzYyZjU5OGQwNTk0NGRiODZiZjVjYTk3ODA0NzYwOCIsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJodHRwczovL3BsYXRmb3JtLnJpbmdjZW50cmFsLmNvbS9yZXN0YXBpL29hdXRoL3Rva2VuIiwic3ViIjoiMzkxNzY1ODAzNiIsImlzcyI6Imh0dHBzOi8vcGxhdGZvcm0ucmluZ2NlbnRyYWwuY29tIiwiZXhwIjozODYxMjY5MzUxLCJpYXQiOjE3MTM3ODU3MDQsImp0aSI6InVMU0RPVTNtU1htYmNxYmdPcmNuY3cifQ.NsRk3UrI9jxYzTDfRE9aHk8Ckz4lzcsLK8aHHZSSxYEex9XcCoJjPFP9QKvsaJS2gQ6b8XbwUL2mKPCANeJ9eavmyOQzyvsnuICi4auxHrCh-VCP6_c1TenowEqOhN73fCmgFaEOOKd8UFVn1vAenmeHMoF0oUC8Gd1WIN6_2p2ukzGCQ7DBaOqYTmjCtIm7ACqL7qI0DPwPTeb9dO2qGYMORcOEM7Me_aSH81_LIpZ194X0GnYUI8ojKBpaqAMrXXveaO1UM29YKPKqW9GNPWs0HUQLejkZQaBWq2XNPapIbC7Zhqa_7nLcan8MdSWKokaGswcGLLYIKC5LRJJw-Q']);
            $response = $this->readExtensionPhoneNumberDetectSmsFeature($platform, $RECIPIENT, $otp);

            $data['status'] = true;
            $data['otp'] = $otp;
            $data['phoneNumber'] = $RECIPIENT;
            $data['message'] = $response;

            return response()->json($data);
        } catch (\RingCentral\SDK\Http\ApiException $e) {
            return "Unable to authenticate to platform. Check credentials. " . $e->getMessage();
        }
    }

    protected function readExtensionPhoneNumberDetectSmsFeature($platform, $RECIPIENT, $otp)
    {
        $endpoint = "/restapi/v1.0/account/~/extension/~/phone-number";
        $resp = $platform->get($endpoint);
        $jsonObj = $resp->json();

        foreach ($jsonObj->records as $record) {
            foreach ($record->features as $feature) {
                if ($feature == "SmsSender") {
                    return $this->sendSmsMessage($platform, $record->phoneNumber, $RECIPIENT, $otp);
                }
            }
        }

        if (count($jsonObj->records) == 0) {
            return "This user does not own a phone number!";
        } else {
            return "None of this user's phone number(s) has the SMS capability!";
        }
    }

    protected function sendSmsMessage($platform, $fromNumber, $RECIPIENT, $otp)
    {
        try {
            $requestBody = [
                'from' => ['phoneNumber' => $fromNumber],
                'to' => [['phoneNumber' => $RECIPIENT]],
                'text' => "Your OTP is: $otp"
            ];
            $endpoint = "/account/~/extension/~/sms";
            $resp = $platform->post($endpoint, $requestBody);
            $jsonObj = $resp->json();
            $this->checkMessageStatus($platform, $jsonObj->id);
            return "SMS sent. Message id: " . $jsonObj->id;
        } catch (\RingCentral\SDK\Http\ApiException $e) {
            return "Error message: " . $e->getMessage();
        }
    }

    protected function checkMessageStatus($platform, $messageId)
    {
        try {
            $endpoint = "/restapi/v1.0/account/~/extension/~/message-store/" . $messageId;
            $resp = $platform->get($endpoint);
            $jsonObj = $resp->json();
            if ($jsonObj->messageStatus == "Queued") {
                sleep(2);
                $this->checkMessageStatus($platform, $jsonObj->id);
            }
            return "Message status: " . $jsonObj->messageStatus;
        } catch (\RingCentral\SDK\Http\ApiException $e) {
            return "Error message: " . $e->getMessage();
        }
    }

    public function signoutStaffSessionsId(Request $request)
    {
        $data['success'] = true;
        try {
            $user = User::findOrNew($request->staffID);
            if ($user && $user->two_step == 1) {
                $user->google2fa_secret = null;
                $user->save();
            }

            $sessionData = DB::table('sessions')->select('id', 'user_id', 'ip_address', 'user_agent', 'last_activity')
                ->where('id', $request->session_id)
                ->where('user_id', $request->staffID)
                ->first();

            $createLogoutHistory = $this->loginHistoryService->update($request->staffID, $request->session_id);

            $sessionHandler = Session::getHandler();
            $sessionHandler->destroy($sessionData->id);

            $sessionId = session()->getId();

            if ($sessionId != $request->session_id) {
                $data['staff_id'] = $request->staffID;
            }

            $data['message'] = 'Session Logout';

            return response()->json($data);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function verifyPassword(Request $request)
    {

        $rules = [
            'password' => 'required|string|min:8',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ]);
        } else {
            $authUser = Auth::user();
            $user = User::find($authUser->id);
            if (Hash::check($request->password, $user->password)) {
                return response()->json(['success' => true, 'message' => 'Password Verification Successfully']);
            } else {
                return response()->json(['errors' => ['password' => ['Incorrect password.']]]);
            }
        }
    }


    public function requestPasswordVerify(Request $request)
    {

        $rules = [
            'email' => 'required|email|exists:users,email',
        ];

        $CustomMessages = [
            'email.exists' => 'The provided email is invalid..',

        ];

        $validator = Validator::make($request->all(), $rules, $CustomMessages);

        if ($validator->fails()) {
            $errorMessage = $validator->errors()->first('email');
            return Redirect::route('request.password')->with('message', $errorMessage);
        } else {
            $user = User::where('email', $request->email)->first();
            $superAdminId = User::role('Super Admin')->first()->id;
            NotificationHelper::notify('staff', 'User with email address ' . $user->email . ' has requested for new password ', [
                'id' => $user->id,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($request->all())->content(),
                'message' => 'New Password Request',
            ], $superAdminId);

            return Redirect::route('request.password')->with('success', "Your request for the new password has been send. You will be revert back through email in next 24 hours.");
        }
    }

    public function export(Request $request)
    {
        $data['success'] = false;
        $rules = [
            'columns' => 'nullable|array',
            'dateRange' => 'nullable|array',
            'rangeOption' => 'nullable',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $data['errors'] = $validator->errors();
            return response()->json($data, 422);
        }

        try {
            $validatedData = $validator->validated();
            $this->staffExport->columns();

            $data['success'] = true;
            $data['data'] = $this->staffExport->export($validatedData);

            return response()->json($data);
        } catch (Exception $ex) {
            $data['message'] = $ex->getMessage();
            return response()->json($data, $ex->getCode());
        }
    }

    public function dailyStaffPayments(Request $request)
    {
        $userId = $request->id ? $request->id : $request->user()->id;

        //End::Getting Dates for the current month

        //Begin::Payment Graph Begin Data
        $graphPayments = [
            'payments' => 0,
            'failedpayments' => 0,
            'upsell' => 0,
            'newsale' => 0,
            'incomplete' => 0,
            'customers' => 0,
            'spendpercustomer' => 0,
        ];
        //End::Payment Graph Begin Data
        if(!empty($request->startDate) && !empty($request->endDate))  {
            //Begin::Getting Payment Data
            $allPayments = Payment::select('id', 'payment_link_id', 'price', 'status')->whereHas('link', function ($link) use ($request, $userId) {
                $link->where('created_by', $userId);
            })->whereDate('created_at', '>=', $request->startDate)->whereDate('created_at', '<=', $request->endDate)->get();

            $spendpercustomer = Payment::select('id', 'payment_link_id', 'price', 'status')->whereHas('link', function ($link) use ($request, $userId) {
                $link->where('created_by', $userId);
                $link->where('status', 2);
            })->where('status', 1)->whereDate('created_at', '>=', $request->startDate)->whereDate('created_at', '<=', $request->endDate)->get();


            $freshsalesCount = $this->getSaleTypeCountFilter("fresh_sales", $request->startDate, $request->endDate, $userId);
            $upsellCount = $this->getSaleTypeCountFilter("upsell", $request->startDate, $request->endDate, $userId);

            //End::Getting Payment Data

            $graphPayments['newsale'] = $freshsalesCount['count'] ?? 0;
            $graphPayments['upsell'] = $upsellCount['count'] ?? 0;
            $graphPayments['payments'] = number_format($allPayments->where('status', 1)->sum('price'));
            $graphPayments['failedpayments'] = number_format($allPayments->where('status', 2)->sum('price'));
            $graphPayments['incomplete'] = number_format($allPayments->where('status', 3)->sum('price'));
            $graphPayments['customers'] = number_format($allPayments->count('price'));
            $graphPayments['spendpercustomer'] = number_format($spendpercustomer->count() > 0 ? $spendpercustomer->sum('price') / $spendpercustomer->count() : 0);
        }
         //End::Setting Data for the graph

        return response()->json([
            'status' => true,
            'data' => $graphPayments,
        ]);

    }

    public function detailLogin(Request $request)
    {
        $page["title"] = "Login History List - Staff Module";
        //Begin::Columns for Login History Table
        $columns = [
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true, "fixed" => true],
            ["key" => "device", "component" => "DeviceColumn", "text" => "Device", "visible" => true],
            ["key" => "platform", "component" => "PlatformColumn", "text" => "Platform", "visible" => true],
            ["key" => "device_type", "component" => "DeviceTypeColumn", "text" => "Device Type", "visible" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "total_duration", "component" => "DurationColumn", "text" => "Duration", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        //Begin::Login History Records of current User
        $loginHistory = LoginHistory::query()
            ->select('id', 'session_id', 'user_id', 'activity', 'ip_address', 'device', 'device_type', 'platform', 'country', 'browser', 'duration', 'created_at')
            ->where('user_id', $request->id)
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->devicetype), function ($query) use ($request) {
                $query->where('device_type', $request->devicetype);
            })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();
        //End::Login History Records of current User
        $loginHistory->getCollection()->transform(function ($record) {

            if (empty($record->duration)) {
                $currentTime = now();
                $createdTime = Carbon::parse($record->created_at);

                $durationInSeconds = $currentTime->diffInSeconds($createdTime);
                $hours = floor($durationInSeconds / 3600);
                $minutes = floor(($durationInSeconds % 3600) / 60);
                $seconds = $durationInSeconds % 60;

                if ($hours > 0) {
                    $formattedDuration = $hours . ($hours == 1 ? ' hour' : ' hours');
                    if ($minutes > 0) {
                        $formattedDuration .= ', ' . $minutes . ' minutes';
                    }
                } elseif ($minutes > 0) {
                    $formattedDuration = $minutes . ' minutes';
                } else {
                    $formattedDuration = $seconds . ' seconds';
                }

                $record->total_duration = $formattedDuration;
            } else {
                $record->total_duration = $record->duration;
            }
            return $record;
        });


        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;
        $userId = $request->id;

        return Inertia::render('Admin/Staff/LoginHistoryList', [
            'loginHistory' => $loginHistory,
            'columns' => $columns,
            'authUserRole' => $authUserRole,
            'userId' => $userId,
            'page' => $page,
        ]);
    }

    private function getSaleTypeCountFilterById($saleType, $startDate, $endDate, $userId)
    {
        return Payment::query()
            ->selectRaw('SUM(payments.price) as count, link.sale_type as name')
            ->join('payment_links as link', 'payments.payment_link_id', '=', 'link.id')
            ->where('link.sale_type', $saleType)
            ->where('payments.status', 1)
            ->whereDate('payments.created_at', '>=', $startDate)
            ->whereDate('payments.created_at', '<=', $endDate)
            ->where('link.created_by', $userId)
            ->groupBy('link.sale_type')
            ->first();
    }


    private function getSaleTypeCountFilter($saleType, $startDate, $endDate, $userId)
    {
        return Payment::query()
            ->selectRaw('SUM(payments.price) as count, link.sale_type as name')
            ->join('payment_links as link', 'payments.payment_link_id', '=', 'link.id')
            ->where('link.sale_type', $saleType)
            ->where('payments.status', 1)
            ->whereBetween('payments.created_at', [$startDate, $endDate])
            ->where('link.created_by', $userId)
            ->groupBy('link.sale_type')
            ->first();
    }
}
