<?php

namespace App\Http\Controllers\Admin;

use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Coupon;
use App\Models\ProductLog;
use App\Services\ProductService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;

class ProductCatalogController extends Controller
{
    private $productService;

    function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }

    public function list(Request $request)
    {
        $page["title"] = "Products List - ProductModule";

        $columns = array(
            array("key" => "name", "component" => 'NameColumn', "text" => "Name", "visible" => true, "fixed" => true),
            array("key" => "created_at", "component" => 'CreatedAtColumn', "text" => "Created", "visible" => true),
            array("key" => "updated_at", "component" => 'UpdatedAtColumn', "text" => "Updated", "visible" => true),
        );

        $statuses = ['all', 'available', 'archived'];

        $products = Category::query()
            ->select('id', 'custom_id', 'name', 'image', 'created_by', 'updated_by', 'status', 'created_at', 'updated_at', 'deleted_at')
            ->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
                $currentStatus = 0;

                foreach ($statuses as $key => $status) {
                    if ($status == $request->status_all) {
                        $currentStatus = $key;
                    }
                }

                if ($currentStatus == 2) {
                    // Include soft-deleted products (archived) with status 2
                    $query->onlyTrashed();
                } elseif ($currentStatus == 1) {
                    // Include only active products with status 1 only
                    $query->where('status', 1);
                } else {
                    // Include available products (both active and archived) with status 1 and 2
                    $query->withTrashed();
                }
            })
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        $productSums = Category::statusCount($statuses);

        return Inertia::render('Admin/Product/ProductList', [
            'columns' => $columns,
            'page' => $page,
            'products' => collect($productSums)->merge($products),
        ]);
    }

    public function create()
    {
        $page["title"] = "Create Product Catalog - ProductModule";

        return Inertia::render('Admin/Product/Create', [
            'page' => $page,
        ]);
    }

    public function generate(Request $request)
    {
        try {
            $rules = [
                'name' => 'required',

            ];

            $customMessages = [
                'name.required' => 'Product Name is required.',
            ];

            if ($request->hasFile('image')) {
                $rules['image'] = 'image|mimes:jpeg,png,jpg,gif|max:2048';
                $customMessages['image.image'] = 'The image must be an image.';
                $customMessages['image.mimes'] = 'The image must be a file of type: jpeg, png, jpg, gif.';
            }

            $validator = Validator::make($request->all(), $rules, $customMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.products.list')
                    ->withErrors($validator)
                    ->withInput();
            }

            $request['created_by'] = 1;
            $productService = $this->productService->create($request->all());

            if ($productService["success"] == true) {
                $log = array(
                    'product_id' => $productService["id"],
                    'activity' => 'Product created',
                    'type' => 'created',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($productService)->content(),
                );

                ProductLog::create($log);
                NotificationHelper::notify('product', 'Product Added' . ' ' . $request->name, [
                    'id' => $productService["id"],
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($productService)->content(),
                    'message' => 'Product Add Successful',
                ], Auth::user()->id);

                Session::flash('success', true);
                Session::flash('message', 'New Product Catalog Created.');
                return Redirect::route('admin.products.list');
            }
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('admin.products.list');
        }
    }

    public function getProduct($productID)
    {
        $page["title"] = "Edit Product Catalog - ProductModule";

        $data['success'] = true;
        try {
            $product = $this->productService->getProductById($productID);

            return Inertia::render('Admin/Product/Edit', [
                'page' => $page,
                'product' => $product,
            ]);

            // return response()->json([
            //     "product" => $product
            // ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    /*public function update(Request $request, $productID) {
        try {
            $rules = [
                'name' => 'required',

            ];

            $customMessages = [
                'name.required' => 'Product Name is required.',
            ];

            if ($request->hasFile('image')) {
                $rules['image'] = 'image|mimes:jpeg,png,jpg,gif|max:2048';
                $customMessages['image.image'] = 'The image must be an image.';
                $customMessages['image.mimes'] = 'The image must be a file of type: jpeg, png, jpg, gif.';
            }

            $validator = Validator::make($request->all(), $rules, $customMessages);

            if ($validator->fails()) {
                return Redirect::back()->withErrors($validator)->withInput();
            }

            $request['id'] = $productID;
            $request['updated_by'] = 1;
            $productService = $this->productService->updateProduct($request->all());

            if ($productService["success"] == true) {
                Session::flash('success', true);
                Session::flash('message', 'Product Catalog Updated.');
                return Redirect::route('admin.products.list');
            }
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('admin.products.list');
        }
    }*/

    public function update(Request $request, $productID)
    {

        try {
            $rules = [
                'name' => 'required',

            ];

            $customMessages = [
                'name.required' => 'Product Name is required.',
            ];

            if ($request->hasFile('image')) {
                $rules['image'] = 'image|mimes:jpeg,png,jpg,gif|max:2048';
                $customMessages['image.image'] = 'The image must be an image.';
                $customMessages['image.mimes'] = 'The image must be a file of type: jpeg, png, jpg, gif.';
            }

            $validator = Validator::make($request->all(), $rules, $customMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.product.detail', $productID)
                    ->withErrors($validator)
                    ->withInput();
            }

            $request['id'] = $productID;
            $request['updated_by'] = 1;
            $productService = $this->productService->updateProduct($request->all());

            if ($productService["success"] == true) {
                $log = array(
                    'product_id' => $productService["id"],
                    'activity' => 'Product updated',
                    'type' => 'updated',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($productService)->content(),
                );
                ProductLog::create($log);
                NotificationHelper::notify('product', 'Product Updated' . ' ' . $request->name, [
                    'id' => $request['id'],
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($productService)->content(),
                    'message' => 'Product Update Successful',
                ], Auth::user()->id);

                Session::flash('success', true);
                Session::flash('message', 'Product Catalog Updated.');
                return Redirect::route('admin.product.detail', $productID);
            }
        } catch (Exception $ex) {
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            return Redirect::route('admin.product.detail', $productID);
        }
    }

    public function archived($productID)
    {
        $data['success'] = true;
        try {
            $productService = $this->productService->archiveProduct($productID);

            if ($productService["success"] == true) {
                $log = array(
                    'product_id' => $productService["id"],
                    'activity' => 'Product archived',
                    'type' => 'archived',
                    'request' => response()->json(['id' => $productID])->content(),
                    'response' => response()->json($productService)->content(),
                );

                ProductLog::create($log);
                Session::flash('success', true);
                Session::flash('message', $productService["message"]);
                return Redirect::route('admin.products.list');
            }
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function delete($productID)
    {
        $data['success'] = true;
        try {
            $productService = $this->productService->deleteProduct($productID);

            if ($productService["success"] == true) {
                Session::flash('success', true);
                Session::flash('message', $productService["message"]);
                return Redirect::route('admin.products.list');
            }
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function restore($productID)
    {
        $data['success'] = true;
        try {
            $productService = $this->productService->restoreProduct($productID);

            if ($productService["success"] == true) {
                $log = array(
                    'product_id' => $productService["id"],
                    'activity' => 'Product restored',
                    'type' => 'restored',
                    'request' => response()->json(['id' => $productID])->content(),
                    'response' => response()->json($productService)->content(),
                );

                ProductLog::create($log);
                Session::flash('success', true);
                Session::flash('message', $productService["message"]);
                return Redirect::route('admin.products.list');
            }
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function detail($productID)
    {
        $page["title"] = "Product Catalog Detail - ProductModule";

        $data['success'] = true;
        try {
            // $product = $this->productService->getProductById($productID);

            $product = Category::where("id", $productID)->with([
                'productlogs' => function ($logs) {
                    $logs->latest('created_at');
                }
            ])->withTrashed()->first();


            return Inertia::render('Admin/Product/Detail', [
                'product' => $product,
                'productLogs' => $product->productlogs,
                'page' => $page,
            ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function updateMetadata(Request $request, $productID)
    {
        $data['success'] = true;
        try {

            $request['id'] = $productID;
            $request['updated_by'] = 1;
            $productService = $this->productService->updateProductMetadata($request->all());

            if ($productService["success"] == true) {
                $log = array(
                    'product_id' => $productService["id"],
                    'activity' => 'Product updated',
                    'type' => 'updated',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($productService)->content(),
                );
                ProductLog::create($log);

                return Redirect::route('admin.product.detail', $productID);
            }
        } catch (Exception $ex) {
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            return Redirect::route('admin.product.detail', $productID);
        }
    }
}
