<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\CountryCurrencies;
use App\Models\PaymentGateway;
use App\Models\PaymentLink;
use App\Models\Role;
use App\Models\Setting;
use App\Models\Team;
use App\Models\User;
use App\Services\CurrencyService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use App\Services\TokenService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use App\Services\LogService;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

class PaymentLinkController extends Controller
{
    private $tokenService, $logService, $currencyService;

    function __construct(TokenService $tokenService, LogService $logService, CurrencyService $currencyService)
    {
        $this->tokenService = $tokenService;
        $this->logService = $logService;
        $this->currencyService = $currencyService;

        $this->middleware('permission:PaymentLinkGenerator-Create|PaymentLinkGenerator-Edit|PaymentLinkGenerator-View|PaymentLinkGenerator-Delete', ['only' => ['index', 'store']]);
        $this->middleware('permission:PaymentLinkGenerator-Create', ['only' => ['form', 'store']]);
        $this->middleware('permission:PaymentLinkGenerator-Edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:PaymentLinkGenerator-Delete', ['only' => ['destroy']]);
    }

    public function create($customerID = 0)
    {
        $page["title"] = "Generate Link - PaymentModule";

        $currencies = $this->currencyService->getCurrencies();
        $teams = Team::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        $paymentGateway = PaymentGateway::select('id', 'name')->get();
        $customer = (object) [];

        if ($customerID > 0) {
            $customer = User::select('id', 'first_name', 'last_name', 'email', 'phone', 'company', 'address')
                ->where('id', '=', $customerID)->first();
        }

        return Inertia::render('Admin/Payment/LinkEditor', [
            'page' => $page,
            'currencies' => $currencies,
            'teams' => $teams,
            'categories' => $categories,
            'paymentGateway' => $paymentGateway,
            'customer' => $customer,
            'method' => 'add'
        ]);
    }

    public function edit(Request $request)
    {
        $page["title"] = "Edit Link - PaymentModule";

        $currencies = $this->currencyService->getCurrencies();
        $teams = Team::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        $paymentGateway = PaymentGateway::select('id', 'name')->get();
        $paymentLink = PaymentLink::whereId($request->id)->with('categories', 'paymentLinkCustomer')->first();

        return Inertia::render('Admin/Payment/LinkEditor', [
            'page' => $page,
            'currencies' => $currencies,
            'teams' => $teams,
            'categories' => $categories,
            'paymentGateway' => $paymentGateway,
            'paymentLink' => $paymentLink,
            'method' => 'edit'
        ]);
    }

    public function delete(Request $request)
    {

        dd($request);
        $rules = [
            'id' => 'nullable|exists:users,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:users,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        // Define roles to include
        $includedRoles = ['Super Admin'];

        // Retrieve roles that have the 'Payment-Global' permission
        $rolesWithPaymentGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'PaymentLinkGenerator-Global');
        })->pluck('name')->toArray();

        // Combine included roles and roles with 'Payment-Global' permission
        $filteredRoles = array_merge($includedRoles, $rolesWithPaymentGlobalPermission);

        $roles = $filteredRoles;

        if ($request->user()->hasAnyRole($roles)) {
            try {
                if ($request->filled('id')) {
                    $customer = User::findOrFail($request->id);
                    $customer->delete();
                } else if ($request->filled('ids')) {
                    User::destroy($request->ids);
                }

                return response()->json([
                    'success' => true,
                    'message' => "Customer(s) successfully deleted.",
                ], 200);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => "An error occurred while deleting customer(s).",
                ], 500);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => "You are not allowed to perform this operation.",
            ], 403);
        }
    }

    public function update(Request $request, $id)
    {
        $item_price = $request->input('item_price');
        try {
            $rules = [
                'saleType' => 'nullable',
                'item_name' => 'nullable',
                'item_price' => 'nullable|min:0',
                'discount' => [
                    'nullable',
                    function ($attribute, $value, $fail) use ($item_price, $request) {
                        $discountType = $request->discountType ?? null;

                        if ($discountType === 'percent') {
                            $discount_value = (float) $value;
                            $discount_amount = ceil(($discount_value / 100) * $item_price);
                            if ($item_price <= $discount_amount) {
                                $fail("The discount must be less than the Item Price.");
                            }
                        } else {
                            $discount_value = (float) $value;
                            if ($item_price <= $discount_value) {
                                $fail("The discount must be less than the Item Price.");
                            }
                        }
                    },
                ],
                'item_description' => 'nullable',
                'comment' => 'nullable',
                'discountType' => 'nullable',
                'cname' => 'nullable',
                'cnumber' => 'nullable',
                'cemail' => 'nullable|email',
                'caddress' => 'nullable',
                'category' => 'nullable|array',
                'category.*' => 'exists:App\Models\Category,id',
                'paymentGateway' => 'nullable|exists:App\Models\PaymentGateway,id',
                //'paymentGateway' => 'required',
                'team' => 'nullable|exists:App\Models\Team,id',
                'ccompany' => 'nullable|string',
            ];

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return Redirect::route('admin.payment.edit', ['id' => $request->id])
                    ->withErrors($validator)
                    ->withInput();
            }

            $request['created_by'] = Auth::id();
            $token = PaymentLink::where('id', $request->id)->first();
            $tokenService = $this->tokenService->update($request->all(), $request->id, $token->token);

            if ($tokenService > 0) {
                Session::flash('success', true);
                Session::flash('message', 'Payment Link updated.');

                $this->logService->log('payment_link', [
                    'activity' => 'Payment Link Updated.',
                    'loggable_id' => $tokenService,
                    'type' => 'payment_link.update_metadata',
                    'request' => $request,
                    'response' => response()->json($tokenService)->content(),
                    'created_by' => $request->user()->id,
                ]);
            } else {
                Session::flash('success', false);
                Session::flash('message', 'Something went wrong');
            }

            return Redirect::route('admin.payment.linkgenerator.list');
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('admin.payment.edit', ['id' => $request->id]);
        }
    }

    public function updateMetaData(Request $request, $id)
    {
        try {
            $payment = PaymentLink::find($id);

            if ($request->has('metadata')) {
                $payment->metadata = json_encode($request->metadata);
            }

            $payment->save();

            $payment->refresh();

            $payment->metadata = json_decode($payment->metadata);

            return Redirect::route('admin.payment.linkgenerator.detail', ['id' => $payment->id]);
        } catch (Exception $ex) {
            return Redirect::back()->with('error', $ex->getMessage());
        }
    }

    public function generate(Request $request)
    {
        $item_price = $request->input('item_price');
        // $request['currency'] = 170;
        try {
            $rules = [
                'saleType' => 'required',
                'item_name' => 'required',
                'item_price' => 'required|min:0',
                'discount' => [
                    'nullable',
                    function ($attribute, $value, $fail) use ($item_price, $request) {
                        $discountType = $request->discountType ?? null;
                        if ($discountType === 'percent') {
                            $discount_value = (float) $value;
                            $discount_amount = ceil(($discount_value / 100) * $item_price);
                            if ($item_price <= $discount_amount) {
                                $fail("The discount must be less than the Item Price.");
                            }
                        } else {
                            $discount_value = (float) $value;
                            if ($item_price <= $discount_value) {
                                $fail("The discount must be less than the Item Price.");
                            }
                        }
                    },
                ],
                'item_description' => 'nullable',
                'comment' => 'nullable',
                'discountType' => 'nullable',
                'cname' => 'nullable',
                'cnumber' => 'nullable',
                'cemail' => 'nullable|email',
                'caddress' => 'nullable',
                'category' => 'required|array',
                'category.*' => 'exists:App\Models\Category,id',
                'paymentGateway' => 'required|exists:App\Models\PaymentGateway,id',
                //'paymentGateway' => 'required',
                'team' => 'required|exists:App\Models\Team,id',
                'currency' => 'required|exists:country_currencies,id',
                'ccompany' => 'nullable|string',
                'showCoupon' => 'nullable|boolean',
            ];

            $CustomMessages = [
                'item_name.required' => 'Item Name is required.',
                'saleType.required' => 'Sale Type is required.',
                'price.required' => "Price is required.",
                'category.required' => "Select atleast one category.",
                'payment_gateway.required' => "Payment Gateway is required.",
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.payment.generate')
                    ->withErrors($validator)
                    ->withInput();
            }

            //$default_currency = Setting::where("key", '=', "default_currency")->first();
            //$currency = CountryCurrencies::where('aplha_code2',$default_currency->value)->first();
            $request['created_by'] = Auth::id();
            $tokenService = $this->tokenService->create($request->all());

            if ($tokenService > 0) {
                Session::flash('success', true);
                Session::flash('message', 'New Payment link generated.');

                $this->logService->log('payment_link', [
                    'activity' => 'Payment Link Created.',
                    'loggable_id' => $tokenService,
                    'type' => 'payment_link.update_metadata',
                    'request' => $request,
                    'response' => response()->json($tokenService)->content(),
                    'created_by' => $request->user()->id,
                ]);
            } else {
                Session::flash('success', false);
                Session::flash('message', 'Something went wrong');
            }

            return Redirect::route('admin.payment.linkgenerator.list');
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('admin.payment.generate');
        }
    }

    public function list(Request $request)
    {
        $user = Auth::user();
        $page["title"] = "Generate Link - PaymentModule";

        $teams = Team::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Token", "visible" => true, "fixed" => true],
            ["key" => "item_name", "component" => "DescriptionColumn", "text" => "Item", "visible" => true],
            ["key" => "price", "component" => "AmountColumn", "text" => "Price", "visible" => true],
            ["key" => "payment_gateway", "component" => "MerchantColumn", "text" => "Merchant", "visible" => true],
        ];

        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Super Admin'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        $roles = $filteredRoles;

        if ($user->hasRole('Super Admin') || ($user->can('PaymentLinkGenerator-Global'))) {
            $columns[] = ['key' => 'creator.first_name', 'component' => 'BadgeColumn', 'text' => 'Salesperson', 'visible' => true];
        }

        $columns[] = ["key" => "status", "component" => "StatusColumn", "text" => "Status", "visible" => true];
        $columns[] = ["key" => "visited", "component" => "VisitedColumn", "text" => "Visited", "visible" => true];
        $columns[] = ["key" => "comment", "component" => "CommentColumn", "text" => "Comment", "visible" => false];

        // Status { 1=>Sucess, 2=>Active, 3=>Failed, 4=>Refunded }

        $paymentLinksQuery = PaymentLink::select('id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'currency', 'payment_gateway', 'comment', 'status', 'created_by')
            ->with(['gateway:id,name,gateway', 'currencyCountry:id,code,symbol', 'payments:id,payment_link_id', 'creator:id,first_name,last_name'])
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'] . ":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'] . ":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when((!empty($request->amountSearch) && !empty($request->amountSearchBy)), function ($query) use ($request) {
                $query->where('price', $request->amountSearchBy, $request->amountSearch);
            })
            ->when((!empty($request->status)), function ($query) use ($request) {
                $statusMap = [
                    'active' => 1,
                    'succeeded ' => 2,
                    'failed' => 3,
                    'refunded' => 4,
                    'deactivated' => 5,
                    'expired' => 6,
                ];

                $status = $statusMap[$request->status] ?? '';

                $query->where('status', $status);
            });

        if ($user->hasRole('Super Admin')) {
            $paymentLinks = $paymentLinksQuery->with('visitedLinks')->latest('id')->paginate(20);
        } elseif ($user->hasAnyRole($roles)) {

            if ($user->can('PaymentLinkGenerator-Global')) {
                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');

                $paymentLinks = $paymentLinksQuery->whereNotIn('created_by', $superAdminIds)->with('visitedLinks')->latest('id')->paginate(20);

            } else {
                $paymentLinks = $paymentLinksQuery->where('created_by', $user->id)->with('visitedLinks')->latest('id')->paginate(20);
            }

        }

        return Inertia::render('Admin/Payment/Linklist', [
            'columns' => $columns,
            'page' => $page,
            'teams' => $teams,
            'categories' => $categories,
            'paymentLinks' => $paymentLinks
        ]);
    }


    public function detail(Request $request, $id)
    {
        $page["title"] = "Payment Link Detail - PaymentModule";

        $linkData = PaymentLink::where('id', $id)
            ->with([
                'categories:id,name,image',
                'currencyCountry',
                'gateway',
                'teamWise',
                'paymentLinkCustomer',
                'visitedLinks',
                'paymentLogs' => function ($logs) {
                    $logs->latest('created_at');
                }
            ])->first();


        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return Inertia::render('Admin/Payment/LinkGeneratorDetail', [
            'page' => $page,
            'linkData' => $linkData,
            'brand_settings' => $brand_settings,
        ]);
    }

    public function statusChange(Request $request)
    {
        foreach ($request->ids as $id) {
            $paymentLink = PaymentLink::find($id);

            if ($request->status == 1) {
                $paymentLink->valid_till = now()->addHours(48)->timestamp;
            } else if ($request->status == 5) {
                $paymentLink->valid_till = now()->timestamp;
            }

            $paymentLink->status = $request->status;
            $paymentLink->save();
        }
    }
}
