<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\BriefHelper;
use App\Helper\FileHelper;
use App\Helper\LogsHelper;
use App\Helper\PDFHelper;
use App\Http\Controllers\Controller;
use App\Models\BriefActivity;
use App\Models\BriefLink;
use App\Models\SmmBrief;
use App\Models\User;
use App\Models\VisitedBrief;
use App\Services\ApiService;
use App\Services\BinApiService;
use App\Services\BriefActivityService;
use App\Services\IpGeoLocationService;
use App\Services\LogService;
use App\Services\SmmBriefService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class SmmBriefController extends Controller
{
    private $smmBrief, $smmBriefService, $apiService, $briefLink, $ipLocationService, $neutrinoApi, $briefActivityService, $logRepository;
    private $logsDirectory = 'smmBrief', $linkLongDirectory = 'brief_link';

    public function __construct(SmmBrief $smmBrief, SmmBriefService $smmBriefService, ApiService $apiService, BriefLink $briefLink, IpGeoLocationService $ipLocationService,  BinApiService $neutrinoApi, BriefActivityService $briefActivityService, LogService $logRepository)
    {
        $this->smmBrief = $smmBrief;
        $this->smmBriefService = $smmBriefService;
        $this->apiService = $apiService;
        $this->briefLink = $briefLink;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->briefActivityService = $briefActivityService;
        $this->logRepository = $logRepository;
    }

    public function index(Request $request)
    {
        $briefs = $this->smmBriefService->list($request);

        $page = [
            'title' => 'SMM Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'smm.brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'smm.brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "business_name", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ["key" => "industry", "component" => "IndustryColumn", "text" => "Industry", "visible" => false],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        // $briefFormLink = 'brief/form';

        // Log the activity before logging out
        // ActivityLogHelper::staticLogActivity(
        //     'visited-log-brief-page',
        //     'Visited logo brief page'
        // );

        return Inertia::render('Admin/Brief/Brief/SMMList', [
            'briefs' => $briefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }

    public function view(Request $request)
    {
        $isDisabled = true;
        if (Auth::check()) {
            $brief = $this->smmBriefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->smmBriefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }


        return Inertia::render('SmmBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->smmBriefService->fetchByEncodedID($encoded_id, true);

            $htmlContent = Blade::render('brief.smm_download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf'; // Adjust the file name as needed

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new SmmBrief()),
                'activity' => 'SMM Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'smm_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = SmmBrief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }

    public function store(Request $request)
    {
        $rules = [
            'email' => 'required|email|max:255',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:40',
            'business_name' => 'required|string|max:400',
            'industry' => 'required|string|max:100',
            'business_email' => 'required|string|max:255',
            'services' => 'required|string|max:65535',
            'business_goal' => 'required|string|max:400',
            'slogan' => 'required|string|max:400',
            'company_address' => 'required|string|max:400',
            'unique_service' => 'required|string|max:65535',
            'audience' => 'required|string|max:65535',
            'website_link' => 'required|string|max:400',
            'website_hosting' => 'required|string|max:400',
            'competitors_website' => 'required|string|max:65535',
            'facebook_link' => 'required|string|max:400',
            'instagram_link' => 'required|string|max:400',
            'pinterest_link' => 'required|string|max:400',
            'social_media_competitors' => 'required|string|max:65535',

        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Name is required.',
            'business_name.required' => 'Business name is required.',
            'phone.required' => 'Phone number is required.',
            'industry.required' => 'Industry is required.',
            'business_email.required' => 'Business email is required.',
            'services.required' => 'Services is required.',
            'business_goal.required' => 'Business goal is required.',
            'slogan.required' => 'Slogan is required.',
            'company_address.required' => 'Company address is required.',
            'unique_service.required' => 'This field is required.',
            'audience.required' => 'Target audience is required.',
            'website_link.required' => 'Website link is required.',
            'website_hosting.required' => 'Website hosting is required.',
            'competitors_website.required' => 'Competitors Website is required.',
            'facebook_link.required' => 'Facebook link is required.',
            'instagram_link.required' => 'Instagram link is required.',
            'pinterest_link.required' => 'Pinterest_link link is required.',
            'social_media_competitors.required' => 'Social media competitors is required.',
        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'company' => $request->business_name,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }


            $smmBrief = $this->smmBriefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $smmBrief['encoded_id']);

            $logData = [
                'activity' => 'SMM Brief has been created by ' . $smmBrief['email'] . '.',
                'request' => $request,
                'response' => response()->json($smmBrief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $smmBrief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $smmBrief['id'],
                'class_type' => get_class(new SmmBrief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'SMM Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($smmBrief)->content(),
                'response' => 201,
                'code' => response()->json($smmBrief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->briefActivityService->create($request, $smmBrief['id'], 'smm_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if($smmBrief['encoded_id']){
                $foundBrief = $this->briefLink->where('id', $smmBrief['brief_link_id'])->first();
                $foundBrief->customer_id = $request->customer_id;
                $foundBrief->status = "submitted";
                $foundBrief->save();
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if($smmBrief['id']){
                $this->smmBrief->find($smmBrief['id'])
                ->update([
                    'location' => $locationData,
                    'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                ]);
            }

            return redirect()->route('smm.brief.success')->with('success', 'Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function stringToArray($string)
    {
        $array = explode(",", $string);
        return $array;
    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'Brand 10X Brief Form',
            'description' => ['Your response has been recorded.']
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:smm_briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:smm_briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $brief = SmmBrief::findOrFail($request->id);
                $brief->delete();
            } else if ($request->filled('ids')) {
                SmmBrief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'smm_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'smm_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }

    }

    public function briefDetail(Request $request){
        $brief = $this->smmBrief->with('briefLogs:id,id,brief_id,activity,type,created_by,request,response,pin,code,event,created_at', 'briefLink:id,link,brief_type,salesperson_id,status,visited,created_at', 'customer:id,first_name,last_name,email,phone,address')->findOrFail($request->id);

        $relatedBrief = [];

        $succeededPayments = collect();

        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'smm');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                'payments' => function ($query) {
                    $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                    $query->with([
                        'link' => function ($linkQ) {
                            $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                            $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                        }
                    ]);
                    $query->latest('id')->get();
                }
            ])->firstOrFail();

            $customer?->payments->each(function ($payment) use ($succeededPayments) {
                if ($payment->link->status == 2) { // Assuming '2' is the status for successful payments
                    $succeededPayments->push($payment->link);
                }
            });
        }

        $briefDetails = [
            'project_details' => [
                'business_email' => $brief->business_email,
                'business_goal' => $brief->business_goal,
                'business_address' => $brief->company_address,
                'name' => $brief->name,
                'contact' => $brief->phone,
                'email' => $brief->email,
            ],
            'project_requirements' => [
                'industry	' => $brief->industry	,
                'services' => $brief->services,
                'unique_service' => $brief->unique_service,
                'audience' => $brief->audience,
                'website_link' => $brief->website_link,
                'competitors_website' => $brief->competitors_website,
                'facebook_link' => $brief->facebook_link,
                'instagram_link' => $brief->instagram_link,
                'pinterest_link' => $brief->pinterest_link,
                'social_media_competitors' => $brief->social_media_competitors,
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);

        $formSubmissionAccuracy = 100;

        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy -= $errorLogsCount;
        $formSubmissionAccuracy = max($formSubmissionAccuracy, 0);

        $briefActivities = VisitedBrief::where('brief_type', 'smm_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'SMM Brief Detail',
            'heading' => 'SMM Brief Details',
            'route' => 'smm.brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'smm.brief.updateMetadata',
            'deleteRoute' => 'smm.brief.delete',
            'formRoute' => 'frontend.smm.brief.view',
            'activityRoute' => 'smm.brief-detail.activity',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'smm.brief-detail.detail',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'smm.brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'smm.brief-detail.details',
                //     'isActive' => false,
                // ],
            ],
        ];

        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'smm_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'smm_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID'  => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }

    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'SMM Brief Activity',
            'heading' => 'SMM Brief Activities',
            'route' => 'smm.brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'SMM Brief Activity',
        //     'heading' => 'SMM Brief Activities',
        //     'route' => 'smm.brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'smm.brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'smm.brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'smm.brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'smm_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }


    // public function details(Request $request){
    //     $brief = $this->smmBriefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'SMM Brief Detail',
    //         'heading' => 'SMM Brief Details',
    //         'route' => 'smm.brief-detail.detail',
    //         'formRoute' => 'frontend.smm.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'smm.brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'smm.brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'smm.brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'email' => $brief->briefLink->customer->email,
    //                 'business_name' => $brief->business_name,
    //                 'owner\'s_name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'business_industry' => $brief->industry,
    //                 'business_email' => $brief->business_email,
    //                 'business_contact_number' => $brief->phone,
    //                 'business_products_/_services' => $brief->services,
    //                 'business_goal' => $brief->business_goal,
    //                 'business_slogan_or_tagline_or_brand_promise' => $brief->slogan,
    //                 'registered_company_address' => $brief->company_address,
    //                 'why_do_you_think_your_product/service_is_unique?_what_makes_you_different?' => $brief->unique_service,
    //                 'target_audience' => $brief->audience,
    //                 'website_link' => $brief->website_link,
    //                 'website_hosting_provider' => $brief->website_hosting,
    //                 'competitor\'s_website_links_(similar_to_your_business)' => $brief->competitors_website,
    //                 'facebook_page_link_(if applicable)' => $brief->facebook_link,
    //                 'instagram_account_link_(if applicable)' => $brief->instagram_link,
    //                 'pinterest_account_link_(if applicable)' => $brief->pinterest_link,
    //                 'social_media_competitors_links_(similar_to_your_business)' => $brief->social_media_competitors,
    //             ],
    //         ]
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }

    public function updateMetadata(Request $request){
        try {
            $query = $this->smmBrief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = SmmBrief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'smm_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);

        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }
}
