<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Http\Controllers\Controller;
use App\Services\MobileAppBriefService;
use App\Models\BriefLink;
use App\Models\MobileAppBrief;
use App\Services\BriefActivityService;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Http\Request;
use App\Helper\BriefHelper;
use App\Helper\FileHelper;
use App\Services\ApiService;
use App\Helper\LogsHelper;
use App\Helper\PDFHelper;
use App\Models\BriefActivity;
use App\Models\User;
use Inertia\Inertia;
use App\Services\IpGeoLocationService;
use Illuminate\Support\Facades\Auth;
use App\Services\BinApiService;
use App\Models\VisitedBrief;
use App\Services\LogService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;

class MobileAppBriefController extends Controller
{

    private $mobileAppBrief, $mobileAppBriefService, $briefActivityService, $briefLink, $apiService, $ipLocationService, $neutrinoApi, $logRepository;
    private $logsDirectory = 'mobileAppBrief', $linkLongDirectory = 'brief_link';

    public function __construct(MobileAppBrief $mobileAppBrief ,MobileAppBriefService $mobileAppBriefService, BriefLink $briefLink, BriefActivityService $briefActivityService, ApiService $apiService, IpGeoLocationService $ipLocationService, BinApiService $neutrinoApi, LogService $logRepository){
        $this->mobileAppBrief = $mobileAppBrief;
        $this->mobileAppBriefService = $mobileAppBriefService;
        $this->briefLink = $briefLink;
        $this->briefActivityService = $briefActivityService;
        $this->apiService = $apiService;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->logRepository = $logRepository;
    }

    public function index(Request $request)
    {

        $appBriefs = $this->mobileAppBriefService->list($request);

        $page = [
            'title' => 'Mobile App Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'app.brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'app.brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "business_name", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        return Inertia::render('Admin/Brief/Brief/AppList', [
            'briefs' => $appBriefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }

    public function store(Request $request)
    {
        $rules = [
        'email' => 'required|email|max:255',
        'name' => 'required|string|max:255',
        'phone' => 'required|string|max:40',
        'business_name' => 'required|string|max:400',
        'business_address' => 'required|string|max:400',
        'business_phone' => 'required|string|max:40',
        'business_email' => 'required|email|max:255',
        'website_link' => 'required|string|max:400',
        'website_hosting' => 'required|string|max:400',
        'website_domain' => 'required|string|max:255',
        'business_operations' => 'required|string|max:400',
        'app_purpose' => 'required|string|max:255',
        'app_look' => 'required|string|max:255',
        'app_specification' => 'required|string|max:400',
        'is_cms' => 'required|string|max:10',
        'content_restriction' => 'required|string|max:400',
        'users_accomplishment' => 'required|string|max:400',
        'app_design' => 'required|string|max:400',
        'publish_assistance' => 'required|string|max:400',
        'app_pages_section' => 'required|string|max:400',
        'app_reference' => 'required|string|max:400',
        'app_users' => 'required|string|max:400',
        'is_push_notification' => 'required|string|max:10',
        'is_third_party_api' => 'required|string|max:400',
        'app_tagline' => 'required|string|max:400',
        'is_app_logo' => 'required|string|max:10',
        'is_logo_redesign' => 'required|string|max:10',
        'app_font_style' => 'required|string|max:400',
        'app_colors' => 'required|string|max:400',
        'comments' => 'required|string|max:400',
        'brief_link_id' => 'required|integer',
        'customer_id' => 'required|integer',
        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Name is required.',
            'phone.required' => 'Phone number is required.',
            'business_name.required' => 'Business name is required.',
            'business_address.required' => 'Business address is required.',
            'business_phone.required' => 'Business phone number is required.',
            'business_email.required' => 'Business email is required.',
            'website_link.required' => 'Website link is required.',
            'website_hosting.required' => 'Web hosting information is required.',
            'website_domain.required' => 'Website domain is required.',
            'business_operations.required' => 'Business operations details are required.',
            'app_purpose.required' => 'App purpose is required.',
            'app_look.required' => 'App look details are required.',
            'app_specification.required' => 'App specification is required.',
            'is_cms.required' => 'CMS status is required.',
            'content_restriction.required' => 'Content restriction details are required.',
            'users_accomplishment.required' => 'User accomplishment details are required.',
            'app_design.required' => 'App design details are required.',
            'publish_assistance.required' => 'Publish assistance details are required.',
            'app_pages_section.required' => 'App pages section is required.',
            'app_reference.required' => 'App reference is required.',
            'app_users.required' => 'App users details are required.',
            'is_push_notification.required' => 'Push notification status is required.',
            'is_third_party_api.required' => 'Third-party API status is required.',
            'app_tagline.required' => 'App tagline is required.',
            'is_app_logo.required' => 'App logo status is required.',
            'is_logo_redesign.required' => 'Logo redesign status is required.',
            'app_font_style.required' => 'App font style is required.',
            'app_colors.required' => 'App colors are required.',
            'comments.required' => 'Comments are required.',
            'brief_link_id.required' => 'Brief link ID is required.',
            'brief_link_id.integer' => 'Brief link ID must be an integer.',
            'customer_id.required' => 'Customer ID is required.',
            'customer_id.integer' => 'Customer ID must be an integer.',
        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'company' => $request->business_name,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }

            $brief = $this->mobileAppBriefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $brief['encoded_id']);

            $logData = [
                'activity' => 'Mobile App Brief has been created by ' . $brief['email'] . '.',
                'request' => $request,
                'response' => response()->json($brief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['id'],
                'class_type' => get_class(new MobileAppBrief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'Mobile App Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($brief)->content(),
                'response' => 201,
                'code' => response()->json($brief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->briefActivityService->create($request, $brief['id'], 'mobile_app_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if ($brief['encoded_id']) {
                $foundBrief = $this->briefLink->where('id', $brief['brief_link_id'])->first();
                if ($foundBrief) {
                    $foundBrief->customer_id = $request->customer_id;
                    $foundBrief->status = "submitted";
                    $foundBrief->save();
                }
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if ($brief['id']) {
                $this->mobileAppBrief->find($brief['id'])
                    ->update([
                        'location' => $locationData,
                        'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                    ]);
            }

            return redirect()->route('app.brief.success')->with('success', 'Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function briefDetail(Request $request)
    {
        $brief = $this->mobileAppBrief->with('briefLink:id,link,brief_type,salesperson_id,status,visited,created_at', 'customer:id,first_name,last_name,email,phone,address')->findOrFail($request->id);

        $relatedBrief = [];

        $succeededPayments = collect();

        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'mobile_app');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                'payments' => function ($query) {
                    $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                    $query->with([
                        'link' => function ($linkQ) {
                            $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                            $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                        }
                    ]);
                    $query->latest('id')->get();
                }
            ])->firstOrFail();

            $customer?->payments->each(function ($payment) use ($succeededPayments) {
                if ($payment->link->status == 2) {
                    $succeededPayments->push($payment->link);
                }
            });
        }

        $briefDetails = [
            'project_details' => [
                'business_address' => $brief->business_address,
                'business_phone' => $brief->business_phone,
                'business_email' => $brief->business_email,
                'business_operations' => $brief->business_operations,
                'name' => $brief->name,
                'contact' => $brief->phone,
                'email' => $brief->email,
            ],
            'project_requirements' => [
                'website' => $brief->website_link,
                'app_purpose' => $brief->app_purpose,
                'app_tagline' => $brief->app_tagline,
                'app_look' => $brief->app_look,
                'app_design' => $brief->app_design,
                'app_specification' => $brief->app_specification,
                'app_pages_section' => $brief->app_pages_section,
                'app_reference' => $brief->app_reference,
                'app_users' => $brief->app_users,
                'app_font_style' => $brief->app_font_style,
                'app_colors' => $brief->app_colors,
                'is_CMS' => $brief->is_cms,
                'is_push_notification' =>$brief->is_push_notification,
                'is_app_logo' => $brief->is_app_logo,
                'is_logo_redesign' => $brief->is_logo_redesign,
                'is_third_party_api' => $brief->is_third_party_api,
                'content_restriction' => $brief->content_restriction,
                'users_accomplishment' => $brief->users_accomplishment,
                'publish_assistance' => $brief->publish_assistance,
                'comments' => $brief->comments,
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);

        $formSubmissionAccuracy = 100;

        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy -= $errorLogsCount;
        $formSubmissionAccuracy = max($formSubmissionAccuracy, 0);

        $briefActivities = VisitedBrief::where('brief_type', 'logo_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'Mobile App Brief Detail',
            'heading' => 'Mobile App Brief Details',
            'route' => 'app.brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'app.brief.updateMetadata',
            'deleteRoute' => 'app.brief.delete',
            'formRoute' => 'frontend.app.brief.view',
            'activityRoute' => 'app.brief-detail.activity',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'app.brief-detail.detail',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'app.brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'app.brief-detail.details',
                //     'isActive' => false,
                // ]
            ],
        ];

        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'mobile_app_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'mobile_app_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID' => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'Mobile App Brief Questionnaire',
            'description' => ['Your response has been recorded.']
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function updateMetadata(Request $request){
        try {
            $query = $this->mobileAppBrief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = MobileAppBrief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'mobile_app_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->mobileAppBriefService->fetchByEncodedID($encoded_id, true);

            $htmlContent = Blade::render('brief.app_download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf';

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new MobileAppBrief()),
                'activity' => 'Mobile App Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'mobile_app_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = MobileAppBrief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }


    public function view(Request $request)
    {
        $isDisabled = true;

        if (Auth::check()) {
            $brief = $this->mobileAppBriefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->mobileAppBriefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }


        return Inertia::render('AppBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:mobile_app_briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:mobile_app_briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $brief = MobileAppBrief::findOrFail($request->id);
                $brief->delete();
            } else if ($request->filled('ids')) {
                MobileAppBrief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'mobile_app_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'mobile_app_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }
    }

    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'Mobile App Brief Activity',
            'heading' => 'Mobile App Brief Activities',
            'route' => 'app.brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'Mobile App Brief Activity',
        //     'heading' => 'Mobile App Brief Activities',
        //     'route' => 'app.brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'app.brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'app.brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'app.brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'mobile_app_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }


    // public function details(Request $request){
    //     $brief = $this->mobileAppBriefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'Mobile App Brief Detail',
    //         'heading' => 'Mobile App Brief Details',
    //         'route' => 'app.brief-detail.detail',
    //         'formRoute' => 'frontend.app.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'app.brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'app.brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'app.brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'email' => $brief->briefLink->customer->email,
    //                 'client\'s_full_name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //                 'client\'s_phone_number' => $brief->briefLink->customer->phone,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'official_business_name' => $brief->business_name,
    //                 'registered_business_address_(if any)' => $brief->business_address,
    //                 'business_contact_number_(if any)' => $brief->business_phone,
    //                 'business_email_(if any)' => $brief->business_email,
    //                 'current_website_URL_(if any)' => $brief->website_link,
    //                 'current_web_hosting_provider_(if any)' => $brief->website_hosting,
    //                 'current_domain_address_(if any)' => $brief->website_domain,
    //                 'please_describe_your_organization\'s_operations_in_a_few_sentences' => $brief->business_operations,
    //                 'what_will_be_the_primary_purpose_of_your_mobile_app' => explode(',', $brief->app_purpose),
    //                 'what_overall_look,_feel,_and_flow_would_you_like_to_protray_with_your_mobile_app' => explode(',', $brief->app_look),
    //                 'please_provide_detailed_information_and_functional_specifications_that_your_mobile_app_will_provide' => $brief->app_specification,
    //                 'would_you_like_to_avail_CMS_features_for_your_mobile_app' => explode(',', $brief->is_cms),
    //                 'is_there_any_content_restriction_outside_of_the_US?' => $brief->content_restriction,
    //                 'what_goals_do_you_have_for_users_to_accomplish_in_your_mobile_app' => $brief->users_accomplishment,
    //                 'do_you_have_wireframes_or_app_designs_(PSD/AI_files)_ready_for_the_application_or_would_you_like_to_avail_our_services_on_app_design?' => $brief->app_design,
    //                 'would_you_like_to_avail_assistance_to_publish_the_app_on_app_stores_like_google_play_&_apple_app_store?' => $brief->publish_assistance,
    //                 'pages/Section_you_require_(e.g.,_about_us,_services/Products,_contact_us)' => $brief->app_pages_section,
    //                 'do_you_have_any_reference_app(s)_which_work_similar_to_your_requirement?_please_list_the_ones_you_found_closest app_reference' => $brief->app_reference,
    //                 'please_provide_detailed_information_about_the_the_users_who_will_be_using_your_mobile_app' => $brief->app_users,
    //                 'would_you_need_push_notifications_for_your_application is_push_notification' => explode(',', $brief->is_push_notification),
    //                 'would_you_like_to_integrate_or_use_any_third-party_API_services_for_your_application?' => $brief->is_third_party_api,
    //                 'do_you_have_a_tagline/slogan' => $brief->app_tagline,
    //                 'do_you_have_a_high-resolution_logo' => explode(',', $brief->is_app_logo),
    //                 'have_you_considered_redesigning_your_logo' => explode(',', $brief->is_logo_redesign),
    //                 'please_specify_if_you_would_like_to_use_a_specific_font_style' => $brief->app_font_style,
    //                 'are_there_any_colors_that_you_want_to_incorporated_into_the_mobile_app?' => $brief->app_colors,
    //                 'anything_else_that_you_would_like_to_add,_feel_free_to_use_the_space_below' => $brief->comments,
    //             ],
    //         ]
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }
}
