<?php

namespace App\Helper;

use App\Models\Brief;
use App\Models\MobileAppBrief;
use App\Models\Payment;
use App\Models\SemBrief;
use App\Models\SmmBrief;
use App\Models\User;
use App\Models\WebBrief;
use App\Models\VideoBrief;
use Exception;
use Illuminate\Support\Facades\DB;

class BriefHelper {

    /**
     * Get briefs counts for showing on top of list as tabs
     *
     * @param string $activeKey
     * @param array  $activeArray
     * @return \Illuminate\Support\Collection
     */
    public static function getBriefsButtonCount($activeKey, $activeArray)
    {
        $briefsCount = collect([
            'logo_brief' => [
                'title' => 'Logo Brief',
                'count' => Brief::count(),
                'active' => false,
                'route' => 'brief.list'
            ],
            'web_brief' => [
                'title' => 'Web Brief',
                'count' => WebBrief::count(),
                'active' => false,
                'route' => 'web.brief.list'
            ],
            'video_brief' => [
                'title' => 'Video Brief',
                'count' => VideoBrief::count(),
                'active' => false,
                'route' => 'video.brief.list'
            ],
            'smm_brief' => [
                'title' => 'SMM Brief',
                'count' => SmmBrief::count(),
                'active' => false,
                'route' => 'smm.brief.list'
            ],
            'sem_brief' => [
                'title' => 'SEM Brief',
                'count' => SemBrief::count(),
                'active' => false,
                'route' => 'sem.brief.list'
            ],
            'mobile_app_brief' => [
                'title' => 'Mobile App Brief',
                'count' => MobileAppBrief::count(),
                'active' => false,
                'route' => 'app.brief.list'
            ]
        ]);

        if ($briefsCount->has($activeKey)) {
            $briefsCount[$activeKey] = $activeArray;
        }

        return $briefsCount;
    }

    /**
     * Get related briefs for the same customer from all types of briefs.
     *
     * @param mixed $customerId
     * @return \Illuminate\Support\Collection
     */
    public static function getRelatedBriefs($customerId, $brief_id, $briefType = null)
    {
        $briefTypes = [
            'logo' => Brief::class,
            'web' => WebBrief::class,
            'video' => VideoBrief::class,
            'smm' => SmmBrief::class,
            'sem' => SemBrief::class,
            'mobile_app' => MobileAppBrief::class,
        ];

        $allBriefs = collect();

        foreach ($briefTypes as $type => $model) {
            $briefs = self::getBriefsByType($model, $customerId, $type);

            // Exclude the brief if the type matches the given briefType
            if ($type === $briefType) {
                $briefs = $briefs->reject(function ($brief) use ($brief_id) {
                    return $brief->id == $brief_id;
                });
            }

            $allBriefs = $allBriefs->merge($briefs);
        }

        return $allBriefs->sortByDesc(function ($item) {
            return $item->created_at;
        })->values();
    }

    /**
     * Get related briefs of the customer from all types of briefs.
     *
     * @param mixed $customerId
     * @return \Illuminate\Support\Collection
     */
    public static function getCustomerRelatedBriefs($customerId)
    {
        $briefTypes = [
            'logo' => Brief::class,
            'web' => WebBrief::class,
            'video' => VideoBrief::class,
            'smm' => SmmBrief::class,
            'sem' => SemBrief::class,
            'mobile_app' => MobileAppBrief::class,
        ];

        $allBriefs = collect();

        foreach ($briefTypes as $type => $model) {
            $briefs = self::getBriefsByType($model, $customerId, $type);
            $allBriefs = $allBriefs->merge($briefs);
        }

        return $allBriefs->sortByDesc(function($item) {
            return $item->created_at;
        })->values();
    }

    /**
     * Get briefs of a specific type for a given customer.
     *
     * @param string $model
     * @param mixed $customerId
     * @param string $briefType
     * @return \Illuminate\Support\Collection
     */
    protected static function getBriefsByType($model, $customerId, $briefType)
    {

        $directories = [
            'logo' => 'brief',
            'web' => 'webBrief',
            'video' => 'videoBrief',
            'smm' =>'smmBrief',
            'sem' => 'semBrief',
            'mobile_app' => 'mobileAppBrief',
        ];

        return $model::where('customer_id', $customerId)
            ->with([
                'briefLink' => function ($link) {
                    $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                        ->with([
                            'salesperson:id,first_name,last_name',
                            'customer:id,first_name,last_name,phone,email,company,stripe_customer_id'
                        ]);
                },
            ])->get()->map(function ($item) use ($briefType, $directories) {
                 //condition should be applied for responsible parties and customer's payments

                $totalSum = Payment::where('status', 1)->where('id', $item->customer->id)->sum('price');
                $activities = LogsHelper::getAllLogs($item->id, $directories[$briefType]);

                $responsiblePartiesIds = [];
                foreach($activities['logs'] as $activity){
                    $responsiblePartiesIds[] = $activity['created_by'];
                }


                $responsiblePartiesIds = array_unique($responsiblePartiesIds);
                rsort($responsiblePartiesIds);

                $responsibleParties = collect();


                if (!empty($responsiblePartiesIds)) {
                    $responsibleParties = User::whereIn('id', $responsiblePartiesIds)
                        ->select('id', 'first_name', 'last_name', 'email', 'image')
                        ->orderByRaw("FIELD(id, " . implode(',', array_map('intval', $responsiblePartiesIds)) . ")")         
                        ->get();
                }

                $item->total_sum = $totalSum;
                $item->responsible_parties = $responsibleParties;
                $item->brief_type = $briefType;
                return $item;
            });
    }

    /**
     * Update metadata for a brief
     *
     * @param mixed $briefId
     * @param array $metadata
     * @throws \Exception
     */
    public static function updateMetadata($query, $metadata)
    {
        try {
            if($query->briefData()){

                $query->briefData()->delete();

                foreach ($metadata as $data) {
                    $query->briefData()->create([
                        'key' => $data['key'],
                        'value' => $data['value'],
                    ]);
                }
            }
        } catch (Exception $e) {
            throw new Exception('Failed to update metadata');
        }
    }
}
