<?php

namespace PubNub;

use PubNub\Exceptions\PubNubBuildRequestException;


class PubNubUtil
{
    /**
     * WARNING: query elements should be already encoded inside Endpoint::customParams() method if required
     *
     * @param string $basePath generated by BasePathManager
     * @param string $path
     * @param string[string] $params query elements
     * @return string url
     */
    public static function buildUrl($basePath, $path, $params)
    {
        return $basePath . $path . "?" . static::joinQuery($params);
    }

    public static function urlWrite($value)
    {
        return static::urlEncode(static::writeValueAsString($value));
    }

    public static function urlEncode($value)
    {
        return rawurlencode($value);
    }

    public static function writeValueAsString($value)
    {
        if (gettype($value) == 'string') {
            return "\"" . $value . "\"";
        } else {
            $res = json_encode($value);

            if (json_last_error()) {
                throw new PubNubBuildRequestException("Value serialization error: " . json_last_error_msg());
            }

            return $res;
        }
    }

    public static function joinQuery($queryElements)
    {
        $keyValues = [];

        foreach ($queryElements as $key => $val) {
            $keyValues[] = "$key=$val";
        }

        return join("&", $keyValues);
    }

    /**
     * @param array $channels
     * @return string
     */
    public static function joinChannels($channels)
    {
        if (count($channels) == 0) {
            return ",";
        } else {
            $encodedChannels = [];

            foreach ($channels as $ch) {
                $encodedChannels[] = urlencode($ch);
            }

            return join(",", $encodedChannels);
        }
    }

    public static function joinItems($items)
    {
        return join(",", $items);
    }


    /**
     * @param $existingItems
     * @param $newItems
     * @return array
     */
    public static function extendArray($existingItems, $newItems)
    {
        if (is_array($newItems)) {
            return array_merge($existingItems, $newItems);
        } else {
            return array_merge($existingItems, static::splitItems($newItems));
        }
    }

    /**
     * @param string $itemsString
     * @return array
     */
    public static function splitItems($itemsString)
    {
        if (strlen((string)$itemsString) === 0) {
            return [];
        } else {
            return explode(",", $itemsString);
        }
    }

    public static function uuid()
    {
        if (function_exists('com_create_guid') === true) {
            return trim(com_create_guid(), '{}');
        } else {
            return sprintf(
                '%04X%04X-%04X-%04X-%04X-%04X%04X%04X',
                mt_rand(0, 65535),
                mt_rand(0, 65535),
                mt_rand(0, 65535),
                mt_rand(16384, 20479),
                mt_rand(32768, 49151),
                mt_rand(0, 65535),
                mt_rand(0, 65535),
                mt_rand(0, 65535)
            );
        }
    }

    public static function preparePamParams($params)
    {
        $sortedParams = $params;
        ksort($sortedParams);
        $stringifiedArguments = "";
        $index = 0;

        foreach ($sortedParams as $key => $value) {
            if (is_bool($value)) {
                $value = $value ? "true" : "false";
            }

            if ($index === count($sortedParams) - 1) {
                $stringifiedArguments .= ($key . "=" . self::pamEncode($value));
            } else {
                $stringifiedArguments .= ($key . "=" . self::pamEncode($value) . "&");
            }

            $index++;
        }

        return $stringifiedArguments;
    }

    public static function pamEncode($url)
    {
        $encoded = self::urlEncode($url);

        if (!empty($encoded)) {
            $encoded = str_replace("*", "%2A", $encoded);
            $encoded = str_replace("!", "%21", $encoded);
            $encoded = str_replace("'", "%27", $encoded);
            $encoded = str_replace("(", "%28", $encoded);
            $encoded = str_replace(")", "%29", $encoded);
            $encoded = str_replace("[", "%5B", $encoded);
            $encoded = str_replace("]", "%5D", $encoded);
            $encoded = str_replace("~", "%7E", $encoded);
        }

        return $encoded;
    }

    public static function signSha256($secret, $signInput)
    {
        $result = strtr(base64_encode(hash_hmac(
            'sha256',
            self::convertIso8859ToUtf8($signInput),
            self::convertIso8859ToUtf8($secret),
            true
        )), '+/', '-_');

        return $result;
    }

    public static function fetchPamPermissionsFrom($jsonInput)
    {
        $r = null;
        $w = null;
        $m = null;
        $d = null;
        $g = null;
        $u = null;
        $j = null;
        $ttl = null;

        if (array_key_exists('r', $jsonInput)) {
            $r = $jsonInput['r'] === 1;
        }

        if (array_key_exists('w', $jsonInput)) {
            $w = $jsonInput['w'] === 1;
        }

        if (array_key_exists('m', $jsonInput)) {
            $m = $jsonInput['m'] === 1;
        }

        if (array_key_exists('d', $jsonInput)) {
            $d = $jsonInput['d'] === 1;
        }

        if (array_key_exists('g', $jsonInput)) {
            $g = $jsonInput['g'] === 1;
        }

        if (array_key_exists('u', $jsonInput)) {
            $u = $jsonInput['u'] === 1;
        }

        if (array_key_exists('j', $jsonInput)) {
            $j = $jsonInput['j'] === 1;
        }

        if (array_key_exists('ttl', $jsonInput)) {
            $ttl = (int) $jsonInput['ttl'];
        }

        return [$r, $w, $m, $d, $g, $u, $j, $ttl];
    }

    public static function isAssoc($arr)
    {
        if (!is_array($arr)) {
            return false;
        }

        return array_keys($arr) !== range(0, count($arr) - 1);
    }

    /**
     * Tests if given string ends with the specified suffix.
     *
     * @param string $string
     * @param string $suffix
     * @return bool
     */
    public static function stringEndsWith($string, $suffix)
    {
        $str_len = strlen($string);
        $suffix_len = strlen($suffix);
        if ($suffix_len > $str_len) {
            return false;
        }
        return substr_compare($string, $suffix, $str_len - $suffix_len, $suffix_len) === 0;
    }

    public static function tokenEncode($token)
    {
        return str_replace('+', '%20', urlencode($token));
    }

    public static function convertIso8859ToUtf8($iso_string)
    {
        $iso_string .= $iso_string;
        $len = strlen($iso_string);

        for ($i = $len >> 1, $j = 0; $i < $len; ++$i, ++$j) {
            switch (true) {
                case $iso_string[$i] < "\x80":
                    $iso_string[$j] = $iso_string[$i];
                    break;
                case $iso_string[$i] < "\xC0":
                    $iso_string[$j] = "\xC2";
                    $iso_string[++$j] = $iso_string[$i];
                    break;
                default:
                    $iso_string[$j] = "\xC3";
                    $iso_string[++$j] = chr(ord($iso_string[$i]) - 64);
                    break;
            }
        }

        return substr($iso_string, 0, $j);
    }
}
