/**
 * Interface for a single form step in the multi-step form.
 *
 * @interface FormStep
 * @property {number} stepNo - The step number in the form.
 * @property {string} title - The title of the form step.
 * @property {string} description - A brief description of the form step.
 */

/**
 * Interface for individual form fields in the Web Design Questionnaire.
 *
 * @interface FormField
 * @property {string} name - The unique identifier for the form field.
 * @property {string} label - The label text displayed for the form field.
 * @property {string} type - The type of input field (e.g., 'text', 'email', 'radio', 'checkbox').
 * @property {string} placeholder - The placeholder text for the input field.
 * @property {string} inputClass - CSS classes applied to the input field.
 * @property {string} description - The description for the input field
 *  @property {boolean} required - Whether the form field is mandatory.
 * @property {number} step - Indicates the step number in which the form field appears.
 * @property {Array<{ label: string, value: string }>} [options] - Applicable for 'radio' and 'checkbox' types, contains label-value pairs for selection (optional).
 * @property {Object} [validation] - Validation rules for input fields (optional).
 * @property {RegExp} [validation.pattern] - Regular expression for pattern matching (optional).
 * @property {string} [validation.errorMessage] - Custom error message shown when validation fails (optional).
 */

/**
 * Interface for the form header section in the Web Design Questionnaire.
 *
 * @interface FormHeader
 * @property {string} heading - The main heading/title of the form.
 * @property {string[]} description - An array of description texts providing context for the form.
 * @property {string[]} notes - Additional notes or instructions for users, typically appearing below the description.
 */


export const LogoBriefData = {
    formHeader: {
        heading: 'Logo Brief Questionnaire',
        description: [
            'Please take your time on the submission below. The more detailed, accurate and extensive information provided, the higher quality logo design concepts you will receive.',
            'We will be needing your contact details to deliver your completed Logo work and/or contact you if we have any questions about your new logo concepts.',
        ],
        notes:  [' Your contact information is safe with us and are never shared with any third party.',]
    },
    formFields: [
        {
            name: 'email',
            label: 'Email',
            type: 'text',
            placeholder: 'Enter Your Email',
            inputClass: 'input email-input',
            required: true,
            purpose: 'email',
        },
        {
            name: 'name',
            label: 'Name',
            type: 'text',
            placeholder: 'Enter Your Name',
            inputClass: 'input name-input',
            required: true,
        },
        {
            name: 'designation',
            label: 'Designation (CEO, Owner, Manager etc)',
            type: 'text',
            placeholder: 'Enter Your Designation',
            inputClass: 'input designation-input',
            required: true,
        },
        {
            name: 'phone',
            label: 'Mobile No',
            type: 'text',
            placeholder: 'Enter Your Mobile Number',
            inputClass: 'input phone-input',
            required: true,
        },
        {
            name: 'business_name',
            label: 'Exact Logo Wording / Business Name',
            type: 'text',
            placeholder: 'Enter Your Business Name',
            inputClass: 'input business-name-input',
            required: true,
        },
        {
            name: 'slogan',
            label: "Tagline / Slogan (optional)",
            type: 'text',
            placeholder: "Enter Business's Slogan",
            inputClass: 'input slogan-input',
        },
        {
            name: 'industry',
            label: 'Business Nature / Industry',
            type: 'select',
            options: [
                { value: null, label: 'Select Your Industry', disabled: true },
                { value: 'accounting_finance', label: 'Accounting & Finance' },
                { value: 'adventure_outdoor', label: 'Adventure & Outdoor' },
                { value: 'agriculture_farming', label: 'Agriculture & Farming' },
                { value: 'art_photography', label: 'Art & Photography' },
                { value: 'attorney_legal', label: 'Attorney & Legal' },
                { value: 'automotive', label: 'Automotive' },
                { value: 'aviation_marine', label: 'Aviation & Marine' },
                { value: 'business_consulting', label: 'Business & Consulting' },
                { value: 'church_religious', label: 'Church & Religious' },
                { value: 'cleaning_services', label: 'Cleaning Services' },
                { value: 'communications', label: 'Communications' },
                { value: 'community_foundations', label: 'Community & Foundations' },
                { value: 'computer_networking', label: 'Computer & Networking' },
                { value: 'construction', label: 'Construction' },
                { value: 'cosmetics_jewelry', label: 'Cosmetics & Jewelry' },
                { value: 'design_creative_services', label: 'Design & Creative Services' },
                { value: 'eco_friendly_green', label: 'Eco-Friendly/Green' },
                { value: 'entertainment_media', label: 'Entertainment & Media' },
                { value: 'fashion_apparel', label: 'Fashion & Apparel' },
                { value: 'fitness_physical_training', label: 'Fitness & Physical Training' },
                { value: 'food_beverages', label: 'Food & Beverages' },
                { value: 'furniture_interior', label: 'Furniture & Interior' },
                { value: 'games', label: 'Games' },
                { value: 'healthcare_physician', label: 'Healthcare & Physician' },
                { value: 'general', label: 'General' },
                { value: 'illustrative', label: 'Illustrative' },
                { value: 'industrial', label: 'Industrial' },
                { value: 'internet_web', label: 'Internet & Web' },
                { value: 'insurance', label: 'Insurance' },
                { value: 'landscaping_yard_care', label: 'Landscaping & Yard Care' },
                { value: 'logistics_transportation', label: 'Logistics & Transportation' },
                { value: 'mortgage_real_estate', label: 'Mortgage & Real Estate' },
                { value: 'nutrition_lifestyle', label: 'Nutrition & Lifestyle' },
                { value: 'pets_animals', label: 'Pets & Animals' },
                { value: 'pharmaceutical_bio_tech', label: 'Pharmaceutical & Bio Tech' },
                { value: 'pool_hot_tub', label: 'Pool & Hot Tub' },
                { value: 'printing_publishing', label: 'Printing & Publishing' },
                { value: 'restaurant', label: 'Restaurant' },
                { value: 'retail', label: 'Retail' },
                { value: 'security', label: 'Security' },
                { value: 'services', label: 'Services' },
                { value: 'spa_salon', label: 'Spa & Salon' },
                { value: 'sports', label: 'Sports' },
                { value: 'staffing_recruiting', label: 'Staffing & Recruiting' },
                { value: 'technology', label: 'Technology' },
                { value: 'travel_hospitality', label: 'Travel & Hospitality' },
                { value: 'oil_gas', label: 'Oil & Gas' },
                { value: 'others', label: 'Others' },
            ],
            required: true,
        },
        {
            name: 'business_description',
            label: 'Business Description',
            type: 'text',
            placeholder: 'Enter Business Description',
            inputClass: 'input business-descripiton-input',
            required: true,
        },
        {
            name: 'services',
            label: 'Business Products / Services',
            type: 'text',
            placeholder: 'Enter Services',
            inputClass: 'input services-input',
            required: true,
        },
        {
            name: 'audience',
            label: 'Target Audience or Customers you want to target?',
            type: 'text',
            placeholder: 'Enter Targeted Audience',
            inputClass: 'input audience-input',
        },
        {
            name: 'website',
            label: 'Business Website',
            type: 'text',
            placeholder: 'Enter Business Website',
            inputClass: 'input website-input',
        },
        {
            name: 'competitor',
            label: 'Competitors or Similar Businesses (for reference purpose)',
            type: 'text',
            placeholder: "Enter Competitors' Name",
            inputClass: 'input competitor-input',
        },
        {
            name: 'logo_title',
            label: 'How do you want your logo to be written?',
            type: 'checkbox',
            options: [
                {value: 'as_it_is', text: 'As it is (as shared above)'},
                {value: 'capital_letters', text: 'In CAPITAL LETTERS'},
                {value: 'small_letters', text: 'In small letters'},
                {value: 'open_to_ideas', text: 'Open to ideas'},
            ],
        },
        {
            name: 'logo_idea',
            label: 'Please share if you already have an idea or concept of how you want your logo to be designed',
            type: 'text',
            placeholder: "Enter Logo's concept",
            inputClass: 'input logo-idea-input',
        },
        {
            name: 'logo_message',
            label: 'Message, which should be communicated through your Logo?',
            type: 'text',
            placeholder: "Enter Logo's Message",
            inputClass: 'input logo-message-input',
        },
        {
            name: 'logo_graphical_element',
            label: 'Any specific type of graphical element would you want to see in your logo?',
            type: 'text',
            placeholder: "Enter Logo's Graphical Element",
            inputClass: 'input logo-graphical-element-input',
        },
        {
            name: 'logo_customer_likes',
            label: 'Logos & Elements, The customer likes (Do´s)',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input logo-customer-likes-input',
        },
        {
            name: 'logo_customer_dont_likes',
            label: "Logos & Elements, The customer don't likes (Dont´s)",
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input logo-customer-dont-likes-input',
        },
        {
            name: 'font_base_logo',
            label: 'Font Based Logo (Select your preference)',
            type: 'checkbox',
            imageOptions: true,
            options: [
                {value: 'just_font', text: 'Just font (Just font without any symbolic intervention).', image: 'backend/images/design-brief/image-1.jpg'},
                {value: 'handmade', text: 'Handmade (A calligraphic, handwritten or script font).', image: 'backend/images/design-brief/image-2.jpg'},
                {value: 'initials', text: 'Initials (Monogram with company name initials).', image: 'backend/images/design-brief/image-3.jpg'},
            ],
        },
        {
            name: 'graphic_base_logo',
            label: 'Graphic Based Logo (Select your preference)',
            type: 'checkbox',
            imageOptions: true,
            options: [
                {value: 'abstract_graphic', text: 'Abstract graphic (A synthetic symbol that represents your Company in a subtle way).', image: 'backend/images/design-brief/image-4.jpg'},
                {value: 'silhouette', text: 'Silhouette (A detailed illustrated silhouette).', image: 'backend/images/design-brief/image-5.jpg'},
                {value: 'illustrated_symbol', text: 'Illustrated symbol (An illustrated symbol that clearly represents an element).', image: 'backend/images/design-brief/image-6.jpg'},
                {value: 'emblem_crests', text: 'Emblem and Crests (Enclosed logo with just text or including graphics).', image: 'backend/images/design-brief/image-7.jpg'},
                {value: 'mascot', text: 'Mascot (A character representing your Company).', image: 'backend/images/design-brief/image-8.jpg'},
            ],
        },
        {
            name: 'logo_color',
            label: 'Any specific type of color would you want to see in your Logo?',
            type: 'text',
            placeholder: 'Eg: Red, Yellow, Black, etc.',
            inputClass: 'input logo-color-input',
            required: true,
        },
        {
            name: 'logo_avoid_color',
            label: 'Avoid these colors',
            type: 'text',
            placeholder: 'Eg: Red, Yellow, Black, etc.',
            inputClass: 'input logo-avoid-color-input',
         },
        {
            name: 'logo_font',
            label: 'What type of FONT style you want to see in your Logo?',
            type: 'text',
            placeholder: 'Eg: Bold, Fancy, Signature, etc.',
            inputClass: 'input logo-font-input',
       },
       {
            name: 'logo_look',
            label: 'Look and Feel of your Logo',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input logo-look-input',
          },
        {
            name: 'logo_use',
            label: 'Where will you use your Logo?',
            type: 'checkbox',
            options: [
                {value: 'websites', text: 'Websites'},
                {value: 'business_stationery_printing', text: 'Business Stationery Printing'},
                {value: 'signage', text: 'Signage / Billboards / Advertisement'},
                {value: 'social_media', text: 'Online Social Media'},
                {value: 'tshirt_printing', text: 'T-Shirt Printing'},
                {value: 'car_wrap', text: 'Car Wrap / Vinyls'},
                {value: 'embroidery', text: 'Embroidery'},
                {value: 'packaging', text: 'Packaging'},
                {value: 'app_icon', text: 'App Icon'},
                {value: 'flyer_brochure', text: 'Flyer / Brochure'},
            ],
            required: true,
          },
        {
            name: 'business_growth',
            label: 'How do you want to grow your business? (MCQ)',
            type: 'checkbox',
            options: [
                {value: 'branding_design', text: 'Branding design and their printing like business cards, letterheads, etc'},
                {value: 'website', text: 'Get a website for your business'},
                {value: 'mobile_app', text: 'Get a mobile app for your business'},
                {value: 'digital_marketing', text: 'Grow your business from digital marketing'},
                {value: 'telesales_support', text: 'Get a service of telesales and support professional via chat, call and email for your business'},
                {value: 'print_ad', text: 'Put an ad on magazine and other print media'},
                {value: 'tradeshow_marketing', text: 'Get marketing support for tradeshows'},
                {value: 'other', text: 'Other'},
            ],
          },
        {
            name: 'logo_budget',
            label: 'Your budget for all the things you have selected above: (Numeric Text)',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input logo-budget-input',
          },
        {
            name: 'contact_time',
            label: 'What time would be convenient to contact you on call? Mention your time zone as well',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input contact-time-input',
          },
        {
            name: 'info',
            sectionType: 'heading',
            heading: 'We want to make sure you have taken time to fill out this logo submission completely with detailed, accurate & extensive information',
            description: 'Our designers create the highest quality logo masterpieces for clients that are serious about having a professional logo designed. When we see a submission form that has been rushed, this shows us that a client is not seriously interested and we only provide them with a simple and basic logo.',
          },
        {
            name: 'anything_else',
            label: 'Anything else to tell our Designers?',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input anything-else-input',
          },
        {
            name: 'completion_date',
            label: 'Required Completion Date',
            type: 'date',
            placeholder: 'Enter Completion Date',
            inputClass: 'input completion-date-input',
            required: true,
        },
    ],
    formFieldsData: {
        email: "",
        name: "",
        designation: "",
        phone: "",
        business_name: "",
        slogan: "",
        industry: "",
        business_description: "",
        services: "",
        audience: "",
        website: "",
        competitor: "",
        logo_title: [],
        logo_idea: "",
        logo_message: "",
        logo_graphical_element: "",
        logo_customer_likes: "",
        logo_customer_dont_likes: "",
        font_base_logo: [],
        graphic_base_logo: [],
        logo_color: "",
        logo_avoid_color: "",
        logo_font: "",
        logo_look: "",
        logo_use: [],
        business_growth: [],
        logo_budget: "",
        contact_time: "",
        anything_else: "",
        completion_date: "",
    }
}

export const WebBriefData = {
    formSteps: [
        {
            stepNo:  1,
            title:  "",
            description:  ""
        },
        {
            stepNo:  2,
            title:  "",
            description:  ""
        },
        {
            stepNo:  3,
            title:  "",
            description:  ""
        },
        {
            stepNo:  4,
            title:  "",
            description:  ""
        }
    ],
    formHeader: {
        heading:  'Web Design Questionnaire',
        description:  [
            'All correspondence and documents provided will be treated as confidential between the client and the designer, unless consent has been granted by both parties involved.'
        ],
        notes:  [' Your contact information is safe with us and are never shared with any third party.',]
    },
    formFields: [
        {
            name:  'email',
            label:  'Email',
            type:  'text',
            placeholder:  'Enter Your Email',
            inputClass:  'input email-input',
            required:  true,
            step:  1,
            purpose: 'email',
        },
        {
            name:  'name',
            label:  'Full Name',
            type:  'text',
            placeholder:  'Enter Your Name',
            inputClass:  'input name-input',
            required:  true,
            step:  1,
        },
        {
            name:  'phone',
            label:  'Mobile Number',
            type:  'text',
            placeholder:  'Enter Your Mobile Number',
            inputClass:  'input phone-input',
            required:  true,
            step:  1,
        },
        {
            name:  'business_name',
            label:  'Business Name',
            type:  'text',
            placeholder:  'Enter Your Business Name',
            inputClass:  'input business-name-input',
            required:  true,
            step:  1,
        },
        {
            name:  'business_address',
            label:  "Business Address (If Available)",
            type:  'text',
            placeholder:  "Enter Your Business Address",
            inputClass:  'input business-address-input',
            required:  false,
            step:  1,
        },
        {
            name:  'business_phone',
            label:  'Business Contact Number (If Available)',
            type:  'text',
            placeholder:  "Enter Your Business Contact Number",
            inputClass:  'input business-phone-input',
            required:  false,
            step:  1,
        },
        {
            name:  'business_email',
            label:  'Business Email (If Available)',
            type:  'text',
            placeholder:  'Enter Your Business Email',
            inputClass:  'input business-email-input',
            required:  false,
            step:  1,
            purpose: 'email',
        },
        {
            name:  'social_media_handles',
            label:  'Social Media Handles URL (Those you would like to connect to your website)',
            type:  'text',
            placeholder:  'Enter Your Social Media Handles',
            inputClass:  'input social-media-handles-input',
            required:  false,
            step:  1,
        },
        {
            name:  'website',
            label:  'Current Website URL (applicable if you have a live business website)',
            type:  'text',
            placeholder:  'Enter Your Business Website',
            inputClass:  'input website-input',
            required:  false,
            step:  1,
        },
        {
            name:  'web_hosting',
            label:  'Current Web Hosting Company (if applicable)',
            type:  'text',
            placeholder:  "Enter Website's Hosting Company",
            inputClass:  'input webhosting-input',
            required:  false,
            step:  1,
        },
        {
            name:  'current_domain_address',
            label:  'Current Domain Address (if applicable)',
            type:  'text',
            placeholder:  "Enter Website's Current Domain",
            inputClass:  'input current-domain-input',
            required:  false,
            step:  1,
        },
        {
            name:  'organization_operations',
            label:  'Please describe your organization\'s operations in a few sentences.',
            type:  'text',
            placeholder:  "Enter Organization Steps",
            inputClass:  'input organization-steps-input',
            required:  true,
            step:  2,
        },
        {
            name:  'website_purpose',
            label:  'Purpose of your website:',
            type:  'radio',
            options:  [
                {value:  'business_information', text:  'Relaying information about your business'},
                {value:  'selling_buying', text:  'Selling (or buying) Online'},
                {value:  'welfare', text:  'Non-profit/Welfare'},
                {value:  'personal_blogging', text:  'Personal Blogging'},
                {value:  'portfolio_showcasing', text:  'Portfolio Showcasing'},
            ],
            required:  true,
            step:  2,
        },
        {
            name:  'portray',
            label:  'Overall feeling that you would like to portray with your new website:',
            type:  'checkbox',
            options:  [
                {value:  'corporate', text:  'Corporate'},
                {value:  'fun', text:  'Fun'},
                {value:  'high-tech', text:  'High-tech'},
                {value:  'modern', text:  'Modern'},
                {value:  'classic', text:  'Classis (Vintage)'},
            ],
            required:  true,
            step:  2,
        },
        {
            name:  'audience',
            label:  'Target audience',
            type:  'text',
            placeholder:  "Enter Targeted Audience",
            inputClass:  'input audience-input',
            required:  false,
            step:  2,
        },
        {
            name:  'pages',
            label:  'Pages you require (except the homepage), e.g. About Us, Services/Products, Contact Us',
            type:  'text',
            placeholder:  "Website Pages",
            inputClass:  'input pages-input',
            required:  false,
            step:  2,
        },
        {
            name:  'competitors',
            label:  'Similar businesses/competitors',
            type:  'text',
            placeholder:  "Enter Your Brand's Competitors",
            inputClass:  'input competitors-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website1_url',
            label:  'Website #1 URL',
            type:  'text',
            placeholder:  "Enter Website's #1 URL",
            inputClass:  'input website-url-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website1_comments',
            label:  'Website # 1 Comments',
            type:  'text',
            placeholder:  "Enter Website # 1 Comments",
            inputClass:  'input website-comment-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website2_url',
            label:  'Website #2 URL',
            type:  'text',
            placeholder:  "Enter Website's #2 URL",
            inputClass:  'input website-url-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website2_comments',
            label:  'Website # 2 Comments',
            type:  'text',
            placeholder:  "Enter Website # 2 Comments",
            inputClass:  'input website-comment-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website3_url',
            label:  'Website #3 URL',
            type:  'text',
            placeholder:  "Enter Website's #3 URL",
            inputClass:  'input website-url-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website3_comments',
            label:  'Website # 3 Comments',
            type:  'text',
            placeholder:  "Enter Website # 3 Comments",
            inputClass:  'input website-comment-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website_features',
            label:  'Any specific features/plugins that you plan to incorporate into your website?',
            type:  'text',
            placeholder:  "Enter Website Feature",
            inputClass:  'input feature-input',
            required:  false,
            step:  2,
        },
        {
            name:  'maintenance_team',
            label:  'Do you plan to hire a maintenance team to update your website',
            type:  'text',
            placeholder:  "Enter Your Answer",
            inputClass:  'input maintenance-input',
            required:  false,
            step:  2,
        },
        {
            name:  'website_content',
            label:  'Do you currently have content that you want to use on your new website?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
            ],
            required:  true,
            step:  2,
        },
        {
            name:  'slogan',
            label:  'Do you have a tagline/slogan?',
            type:  'text',
            placeholder:  "Enter Your Business's Slogan",
            inputClass:  'input slogan-input',
            required:  false,
            step:  3,
        },
        {
            name:  'have_logo',
            label:  'Do you have a high-resolution logo?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
            ],
            required:  true,
            step:  3,
        },
        {
            name:  'redesign_logo',
            label:  'Have you considered redesigning your logo?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
            ],
            required:  true,
            step:  3,
        },
        {
            name:  'website_fonts',
            label:  'What fonts do you typically use?',
            type:  'text',
            placeholder:  "Enter Your Preferred Fonts",
            inputClass:  'input fonts-input',
            required:  false,
            step:  3,
        },
        {
            name:  'website_colors',
            label:  'Are there any colors that you want incorporated into the website?',
            type:  'text',
            placeholder:  "Enter Your Preferred Colors",
            inputClass:  'input colors-input',
            required:  false,
            step:  3,
        },
        {
            name:  'anything_else',
            label:  'If there is anything else that you would like to add,  feel free to use the space below',
            type:  'text',
            placeholder:  "Enter Your Answer",
            inputClass:  'input comments-input',
            required:  false,
            step:  4,
        },
    ],
    formFieldsData:  {
        email:  "",
        name:  "",
        phone:  "",
        business_name:  "",
        business_address:  "",
        business_phone:  "",
        business_email:  "",
        social_media_handles:  "",
        website:  "",
        web_hosting:  "",
        current_domain_address:  "",
        organization_operations:  "",
        website_purpose:  "",
        portray:  [],
        audience:  "",
        pages:  "",
        competitors:  "",
        website1_url:  "",
        website1_comments:  "",
        website2_url:  "",
        website2_comments:  "",
        website3_url:  "",
        website3_comments:  "",
        website_features:  "",
        maintenance_team:  "",
        website_content:  "",
        slogan:  "",
        have_logo:  "",
        redesign_logo:  "",
        website_fonts:  "",
        website_colors:  "",
        anything_else: "",
    },
}

export const VideoBriefData = {
    formSteps: [
        {
            stepNo:  1,
            title:  "",
            description:  ""
        },
        {
            stepNo:  2,
            title:  "",
            description:  ""
       },
        {
            stepNo:  3,
            title:  "",
            description:  ""
       },
        {
            stepNo:  4,
            title:  "",
            description:  ""
        }
    ],
    formHeader: {
        heading:  'Video Brief Questionnaire',
        description:  [
            'All correspondence and documents provided will be treated as confidential between the client and the designer, unless consent has been granted by both parties involved.'
        ],
        notes:  [' Your contact information is safe with us and are never shared with any third party.',]
    },
    formFields: [
        {
            name:  'email',
            label:  'Email',
            type: 'text',
            placeholder:  'Enter Your Email',
            inputClass:  'input email-input',
            required:  true,
            step:  1,
            purpose: 'email',
       },
        {
            name:  'name',
            label:  'Client Name',
            type:  'text',
            placeholder:  'Enter Your Name',
            inputClass:  'input name-input',
            required:  true,
            step:  2,
       },
        {
            name:  'phone',
            label:  'Contact Number',
            type:  'text',
            placeholder:  'Enter Your Mobile Number',
            inputClass:  'input phone-input',
            required:  true,
            step:  2,
       },
        {
            name:  'project_name',
            label:  'Name of the Project',
            type:  'text',
            placeholder:  'Enter Your Project Name',
            inputClass:  'input project-name-input',
            required:  true,
            step:  2,
       },
       {
        name: 'industry',
        label: 'Business Industry',
        type: 'select',
        options: [
            { value: null, label: 'Select Your Industry', disabled: true },
            { value: 'accounting_finance', label: 'Accounting & Finance' },
            { value: 'adventure_outdoor', label: 'Adventure & Outdoor' },
            { value: 'agriculture_farming', label: 'Agriculture & Farming' },
            { value: 'art_photography', label: 'Art & Photography' },
            { value: 'attorney_legal', label: 'Attorney & Legal' },
            { value: 'automotive', label: 'Automotive' },
            { value: 'aviation_marine', label: 'Aviation & Marine' },
            { value: 'business_consulting', label: 'Business & Consulting' },
            { value: 'church_religious', label: 'Church & Religious' },
            { value: 'cleaning_services', label: 'Cleaning Services' },
            { value: 'communications', label: 'Communications' },
            { value: 'community_foundations', label: 'Community & Foundations' },
            { value: 'computer_networking', label: 'Computer & Networking' },
            { value: 'construction_real_estate', label: 'Construction & Real Estate' },
            { value: 'cosmetics_jewelry', label: 'Cosmetics & Jewelry' },
            { value: 'entertainment_media', label: 'Entertainment & Media' },
            { value: 'fashion_apparel', label: 'Fashion & Apparel' },
            { value: 'fitness_physical_training', label: 'Fitness & Physical Training' },
            { value: 'food_beverages', label: 'Food & Beverages' },
            { value: 'furniture_interior', label: 'Furniture & Interior' },
            { value: 'healthcare_physician', label: 'Healthcare & Physician' },
            { value: 'logistics_transportation', label: 'Logistics & Transportation' },
            { value: 'nutrition_lifestyle', label: 'Nutrition & Lifestyle' },
            { value: 'pets_animals', label: 'Pets & Animals' },
            { value: 'pharmaceutical_bio_tech', label: 'Pharmaceutical & Bio Tech' },
            { value: 'printing_publishing', label: 'Printing & Publishing' },
            { value: 'retail', label: 'Retail' },
            { value: 'security', label: 'Security' },
            { value: 'sports', label: 'Sports' },
            { value: 'staffing_recruiting', label: 'Staffing & Recruiting' },
            { value: 'technology', label: 'Technology' },
            { value: 'tourism_hospitality', label: 'Tourism & Hospitality' },
            { value: 'oil_gas', label: 'Oil & Gas' },
            { value: 'mortgage_Insurance', label: 'Mortgage & Insurance' },
        ],
            required: true,
            step:  2,
        },
        {
            name:  'website',
            label:  'Current Website Address (If Any)',
            type:  'text',
            placeholder:  'Enter Your Website',
            inputClass:  'input website-input',
            required:  true,
            step:  3,
       },
        {
            name:  'social_media',
            label:  'Current Social Media Profile (If Any)',
            type:  'text',
            placeholder:  'Enter Your Social Media Links',
            inputClass:  'input social-media-input',
            required:  true,
            step:  3,
       },
        {
            name:  'business_description',
            label:  'Business Description',
            type:  'text',
            placeholder:  'Describe Your Business',
            inputClass:  'input business-description-input',
            required:  true,
            step:  3,
       },
        {
            name:  'business_services',
            label:  'Business Products / Services',
            type:  'text',
            placeholder:  'List Your Business Services',
            inputClass:  'input business-services-input',
            required:  true,
            step:  3,
       },
        {
            name:  'audience',
            label:  'Target Audience or Customers your want to Target?',
            type:  'text',
            placeholder:  'Describe Your Target Audience',
            inputClass:  'input audience-input',
            required:  true,
            step:  3,
       },
        {
            name:  'competitor',
            label:  'Competitors or Similar Businesses (For Reference Purpose)',
            type:  'text',
            placeholder:  'Enter Your Answer',
            inputClass:  'input competitor-input',
            required:  true,
            step:  3,
       },
        {
            name:  'u_s_p',
            name:  'u_s_p',
            label:  'How is your company different from your competitors? (Your USP)',
            type:  'text',
            placeholder:  'Enter Your Unique Selling Proposition',
            inputClass:  'input usp-input',
            required:  true,
            step:  3,
       },
        {
            name:  'comments',
            label:  'Additional Comments - eg. My Vision For the Script',
            type:  'text',
            placeholder:  'Enter Any Comments',
            inputClass:  'input comments-input',
            required:  true,
            step:  3,
       },
        {
            name:  'video_purpose',
            label:  'Purpose of the Video',
            type:  'text',
            placeholder:  'Enter the Purpose of the Video',
            inputClass:  'input video-purpose-input',
            required:  true,
            step:  4,
       },
        {
            name:  'video_duration',
            label:  'What is the duration of the video?',
            type:  'text',
            placeholder:  'Enter the Desired Duration of the Video',
            inputClass:  'input video-duration-input',
            required:  true,
            step:  4,
       },
        {
            name:  'reference',
            label:  'Would you like to follow any reference video e.g. character animation, motion graphic or whiteboard?',
            type:  'text',
            placeholder:  'Enter Any References',
            inputClass:  'input reference-input',
            required:  true,
            step:  4,
       },
        {
            name:  'style',
            label:  'Would you like to follow any specific style of characters?',
            type:  'text',
            placeholder:  'Enter the Style You Prefer',
            inputClass:  'input style-input',
            required:  true,
            step:  4,
       },
        {
            name:  'color',
            label:  'Would you like to use any specific color pallets?',
            type:  'text',
            placeholder:  'Enter the Preferred Colors',
            inputClass:  'input color-input',
            required:  true,
            step:  4,
       },
       {
        name: 'voice',
        label: 'Voice-over preference? (Male/female)',
        type: 'checkbox',
        options: [
            {value: 'male', text: 'Male'},
            {value: 'female', text: 'Female'},
        ],
        required:  true,
        step:  4,
        },
        {
            name:  'artist_accent',
            label:  'Voice-over Artist Accent',
            type:  'text',
            placeholder:  'Enter the Preferred Artist Accent',
            inputClass:  'input artist-accent-input',
            required:  true,
            step:  4,
       },
        {
            name:  'animation_style',
            label:  'Animation Style',
            type:  'text',
            placeholder:  'Enter the Preferred Animation Style',
            inputClass:  'input animation-style-input',
            required:  true,
            step:  4,
       },
        {
            name:  'reference_video',
            label:  'Any Reference Video You Like us to use as Inspiration',
            type:  'text',
            placeholder:  'Enter Reference Video URL',
            inputClass:  'input reference-video-input',
            required:  true,
            step:  4,
       },
        {
            name:  'deadline',
            label:  'Are you working with any deadline?',
            type:  'date',
            placeholder:  'Enter the Project Deadline',
            inputClass:  'input deadline-input',
            required:  true,
            step:  4,
       },
        {
            name:  'additional_comments',
            label:  'Additional Comments over the Video',
            type:  'text',
            placeholder:  'Enter Any Additional Comments',
            inputClass:  'input additional-comments-input',
            required:  true,
            step:  4,
        },
    ],
    formFieldsData: {
        email:  "",
        name:  "",
        phone:  "",
        project_name:  "",
        industry:  "",
        website:  "",
        social_media:  "",
        business_description:  "",
        business_services:  "",
        audience:  "",
        competitor:  "",
        u_s_p:  "",
        comments:  "",
        video_purpose:  "",
        video_duration:  "",
        reference:  "",
        style:  "",
        color:  "",
        voice: [],
        artist_accent:  "",
        animation_style:  "",
        reference_video:  "",
        deadline:  "",
        additional_comments:  "",
    }
}

export const SMMBriefData = {
    formHeader: {
        heading: 'Brand 10X Brief Form',
        description: [],
        notes:  []
    },
    formFields: [
        {
            name: 'email',
            label: 'Email',
            type: 'text',
            placeholder: 'Enter Your Email',
            inputClass: 'input email-input',
            required: true,
            purpose: 'email',
        },
        {
            name: 'business_name',
            label: 'Business Name',
            type: 'text',
            placeholder: 'Enter Your Business Name',
            inputClass: 'input business-name-input',
            required: true,
        },
        {
            name: 'name',
            label: "Owner's Name",
            type: 'text',
            placeholder: 'Enter Your Name',
            inputClass: 'input name-input',
            required: true,
        },
        {
            name: 'industry',
            label: 'Business Industry',
            type: 'select',
            options: [
                { value: null, label: 'Select Your Industry', disabled: true },
                { value: 'accounting_finance', label: 'Accounting & Finance' },
                { value: 'adventure_outdoor', label: 'Adventure & Outdoor' },
                { value: 'agriculture_farming', label: 'Agriculture & Farming' },
                { value: 'art_photography', label: 'Art & Photography' },
                { value: 'attorney_legal', label: 'Attorney & Legal' },
                { value: 'automotive', label: 'Automotive' },
                { value: 'aviation_marine', label: 'Aviation & Marine' },
                { value: 'business_consulting', label: 'Business & Consulting' },
                { value: 'church_religious', label: 'Church & Religious' },
                { value: 'cleaning_services', label: 'Cleaning Services' },
                { value: 'communications', label: 'Communications' },
                { value: 'community_foundations', label: 'Community & Foundations' },
                { value: 'computer_networking', label: 'Computer & Networking' },
                { value: 'construction_real_estate', label: 'Construction & Real Estate' },
                { value: 'cosmetics_jewelry', label: 'Cosmetics & Jewelry' },
                { value: 'entertainment_media', label: 'Entertainment & Media' },
                { value: 'fashion_apparel', label: 'Fashion & Apparel' },
                { value: 'fitness_physical_training', label: 'Fitness & Physical Training' },
                { value: 'food_beverages', label: 'Food & Beverages' },
                { value: 'furniture_interior', label: 'Furniture & Interior' },
                { value: 'healthcare_physician', label: 'Healthcare & Physician' },
                { value: 'logistics_transportation', label: 'Logistics & Transportation' },
                { value: 'nutrition_lifestyle', label: 'Nutrition & Lifestyle' },
                { value: 'pets_animals', label: 'Pets & Animals' },
                { value: 'pharmaceutical_bio_tech', label: 'Pharmaceutical & Bio Tech' },
                { value: 'printing_publishing', label: 'Printing & Publishing' },
                { value: 'retail', label: 'Retail' },
                { value: 'security', label: 'Security' },
                { value: 'sports', label: 'Sports' },
                { value: 'staffing_recruiting', label: 'Staffing & Recruiting' },
                { value: 'technology', label: 'Technology' },
                { value: 'tourism_hospitality', label: 'Tourism & Hospitality' },
                { value: 'oil_gas', label: 'Oil & Gas' },
                { value: 'mortgage_Insurance', label: 'Mortgage & Insurance' },
            ],
            required: true,
        },
        {
            name:  'business_email',
            label:  'Business Email',
            type: 'text',
            placeholder:  'Enter Your Business Email',
            inputClass:  'input business-email-input',
            required:  true,
            purpose: 'email',
        },
        {
            name:  'phone',
            label:  'Business Contact Number',
            type:  'text',
            placeholder:  "Enter Your Business Contact Number",
            inputClass:  'input business-phone-input',
            required:  true,
        },
        {
            name: 'services',
            label: 'Business Product / Services:  (Compulsory)',
            type: 'text',
            placeholder: 'Enter Services',
            inputClass: 'input services-input',
            required: true,
        },
        {
            name: 'business_goal',
            label: 'Business Goal:  (Compulsory)',
            type: 'text',
            placeholder: 'Enter Business Goal',
            inputClass: 'input business-goal-input',
            required: true,
        },
        {
            name: 'slogan',
            label: "Brand Slogan or Tagline or Brand Promise:  (Compulsory)",
            type: 'text',
            placeholder: "Enter Business's Slogan",
            inputClass: 'input slogan-input',
            required: true,
        },
        {
            name: 'company_address',
            label: "Registered Company Address: (Compulsory)",
            type: 'text',
            placeholder: "Enter Company Address",
            inputClass: 'input slogan-input',
            required: true,
        },
        {
            name: 'unique_service',
            label: "Why do you think your product/service is unique? What makes you different?",
            type: 'text',
            placeholder: "Enter Your Answer",
            inputClass: 'input unique-service-input',
            required: true,
        },
        {
            name: 'audience',
            label: 'Target Audience',
            type: 'text',
            placeholder: 'Enter Targeted Audience',
            inputClass: 'input audience-input',
            required: true,
        },
        {
            name: 'website_link',
            label: 'Website Link',
            type: 'text',
            placeholder: 'Enter Website Link',
            inputClass: 'input website-link-input',
            required: true,
        },
        {
            name: 'website_hosting',
            label: 'Website Hosting Provider',
            type: 'text',
            placeholder: 'Enter Website Hosting',
            inputClass: 'input website-hosting-input',
            required: true,
        },
        {
            name: 'competitors_website',
            label: 'Competitors website links which are similar with your business',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input competitors-website-input',
            required: true,
        },
        {
            name: 'facebook_link',
            label: 'Facebook Page Link (If they exist)',
            type: 'text',
            placeholder: 'Enter Facebook Page Link',
            inputClass: 'input facebook-link-input',
            required: true,
        },
        {
            name: 'instagram_link',
            label: 'Instagram Account Link (If they exist)',
            type: 'text',
            placeholder: 'Enter Instagram Account Link',
            inputClass: 'input instagram-link-input',
            required: true,
        },
        {
            name: 'pinterest_link',
            label: 'Pinterest Account Link (If they exist)',
            type: 'text',
            placeholder: 'Enter Pinterest Account Link',
            inputClass: 'input pinterest-link-input',
            required: true,
        },
        {
            name: 'social_media_competitors',
            label: 'Social Media Competitors links which are similar with your business',
            type: 'text',
            placeholder: 'Enter Your Answer',
            inputClass: 'input social-media-competitors-input',
            required: true,
        },
    ],
    formFieldsData: {
        email: "",
        business_name: "",
        name: "",
        industry: "",
        business_email: "",
        phone: "",
        services: "",
        business_goal: "",
        slogan: "",
        company_address: "",
        unique_service: "",
        audience: "",
        website_link: "",
        website_hosting: "",
        competitors_website: "",
        facebook_link: "",
        instagram_link: "",
        pinterest_link: "",
        social_media_competitors: "",
    }
}

export const SEMBriefData = {
    formHeader: {
        heading: 'SEM Brief Questionnaire',
        description: [
            'Please take your time on the submission below. We will be needing your contact details to deliver your completed work and/or contact you if we have any questions.',
        ],
        notes:  [' Your contact information is safe with us and are never shared with any third party.',]
    },
    formFields: [
        {
            name: 'email',
            label: 'Email',
            type: 'email',
            placeholder: 'Enter Your Email',
            inputClass: 'input email-input',
            required: true,
        },
        {
            name: 'name',
            label: 'Name',
            type: 'text',
            placeholder: 'Enter Your Name',
            inputClass: 'input name-input',
            required: true,
        },
        {
            name: 'designation',
            label: 'Designation (CEO, Owner, Manager etc)',
            type: 'text',
            placeholder: 'Enter Your Designation',
            inputClass: 'input designation-input',
            required: true,
        },
        {
            name: 'phone',
            label: 'Mobile No',
            type: 'text',
            placeholder: 'Enter Your Mobile Number',
            inputClass: 'input phone-input',
            required: true,
        },
        {
            name: 'business_name',
            label: 'Registered Business Name',
            type: 'text',
            placeholder: 'Enter Your Business Name',
            inputClass: 'input business-name-input',
            required: true,
        },
        {
            name: 'business_idea',
            label: "What is your business idea? (product/service)",
            type: 'text',
            placeholder: "Enter Business idea",
            inputClass: 'input business-idea-input',
            required: true,
        },
        {
            name: 'competitors',
            label: "List all your competitors",
            type: 'text',
            placeholder: "Enter competitors",
            inputClass: 'input competitors-input',
            required: true,
        },
        {
            name: 'product_strengths',
            label: "What are the strengths of your product/service?",
            type: 'text',
            placeholder: "Enter your answer",
            inputClass: 'input product-strengths-input',
            required: true,
        },
        {
            name: 'product_weaknesses',
            label: 'What are the weaknesses of your product/service?',
            type: 'text',
            placeholder: 'Enter your answer',
            inputClass: 'input product-weaknesses-input',
            required: true,
        },
        {
            name: 'actions',
            label: 'What actions do you want visitors to take on your website?',
            type: 'text',
            placeholder: 'Enter your answer',
            inputClass: 'input actions-input',
            required: true,
        },
        {
            name:  'campaign_purpose',
            label:  'What purpose you wish to solve or achieve from this campaign?',
            type:  'radio',
            options:  [
                {value:  'build_brand_awareness', text:  'Build Brand Awareness'},
                {value:  'drive_leads', text:  'Drive Leads'},
                {value:  'content_downloads', text:  'Content Downloads'},
                {value:  'sign_ups', text:  'Sign Ups'},
                {value:  'increase_website_traffic', text:  'Increase Website Traffic'},
                {value:  'generate_revenue', text:  'Generate Revenue'},
            ],
            required:  true,
        },
        {
            name: 'landing_pages',
            label: 'What landing pages do you want to target?',
            type: 'text',
            placeholder: 'Enter your answer',
            inputClass: 'input landing-pages-input',
            required: true,
        },
        {
            name: 'u_s_p',
            label: 'What makes you different from your competitors?',
            type: 'text',
            placeholder: 'Enter your unique selling points',
            inputClass: 'input u-s-p-input',
            required: true,
        },
        {
            name: 'target_places',
            label: 'List the cities or states, or counties that you want to target',
            type: 'text',
            placeholder: 'Enter your answer',
            inputClass: 'input target-places-input',
            required: true,
        },
        {
            name: 'budget',
            label: 'Please specify the amount you wish to spend on this campaign monthly.',
            type: 'checkbox',
            options: [
                {value: 'USD_500', text: 'USD - 500'},
                {value: 'USD_600', text: 'USD - 600'},
                {value: 'USD_700', text: 'USD - 700'},
                {value: 'USD_800', text: 'USD - 800'},
                {value: 'more_than_USD_800', text: 'More than USD - 800'},
            ],
            required: true,
          },
    ],
    formFieldsData: {
        email: "",
        name: "",
        designation: "",
        phone: "",
        business_name: "",
        business_idea: "",
        competitors: "",
        product_strengths: "",
        product_weaknesses: "",
        actions: "",
        campaign_purpose: "",
        landing_pages: "",
        u_s_p: "",
        target_places: "",
        budget: [],
    }
}

export const MobileAppBriefData = {
    formSteps: [
        {
            stepNo:  1,
            title:  "",
            description:  ""
        },
        {
            stepNo:  2,
            title:  "",
            description:  ""
        },
        {
            stepNo:  3,
            title:  "",
            description:  ""
        },
        {
            stepNo:  4,
            title:  "",
            description:  ""
        },
        {
            stepNo:  5,
            title:  "",
            description:  ""
        },
        {
            stepNo:  6,
            title:  "",
            description:  ""
        }
    ],
    formHeader: {
        heading: 'Mobile App Brief Questionnaire',
        description: [
            'Please take your time on the submission below. The more detailed, accurate and extensive information provided, the higher quality product you will receive.',
        ],
        notes:  []
    },
    formFields: [
        {
            name:  'email',
            label:  'Email',
            type:  'text',
            placeholder:  'Enter Your Email',
            inputClass:  'input email-input',
            required:  true,
            step:  1,
            purpose: 'email',
        },
        {
            name:  'name',
            label:  'Client\'s Full Name',
            type:  'text',
            placeholder:  'Enter Your Name',
            inputClass:  'input name-input',
            required:  true,
            step:  2,
        },
        {
            name:  'phone',
            label:  'Client\'s Phone Number',
            type:  'text',
            placeholder:  'Enter Your Mobile Number',
            inputClass:  'input phone-input',
            required:  true,
            step:  2,
        },
        {
            name:  'business_name',
            label:  'Official Business Name',
            type:  'text',
            placeholder:  'Enter Your Business Name',
            inputClass:  'input business-name-input',
            required:  true,
            step:  2,
        },
        {
            name:  'business_address',
            label:  "Registered Business Address (If Any)",
            type:  'text',
            placeholder:  "Enter Your Business Address",
            inputClass:  'input business-address-input',
            description: 'Please write NA in response of this question if there is no registered business address',
            required:  true,
            step:  2,
        },
        {
            name:  'business_phone',
            label:  'Business Contact Number (If Any)',
            type:  'text',
            placeholder:  "Enter Your Business Contact Number",
            inputClass:  'input business-phone-input',
            description: 'Please write NA in response of this question if there is no business contact number',
            required:  true,
            step:  2,
        },
        {
            name:  'business_email',
            label:  'Business Email (If Any)',
            type:  'text',
            placeholder:  'Enter Your Business Email',
            inputClass:  'input business-email-input',
            description: 'Please write NA in response of this question if there is no business email',
            required:  true,
            step:  2,
            purpose: 'email',
        },
        {
            name:  'website_link',
            label:  'Current Website URL (If Any)',
            type:  'text',
            placeholder:  'Enter Your Website URL',
            inputClass:  'input social-media-handles-input',
            description: 'Please write NA in response of this question if there is no current website URL',
            required:  true,
            step:  2,
        },
        {
            name:  'website_hosting',
            label:  'Current Web Hosting Provider (If Any)',
            type:  'text',
            placeholder:  "Enter Website's Hosting Company",
            inputClass:  'input webhosting-input',
            description: 'Please write NA in response of this question if there is no current web hosting provider',
            required:  true,
            step:  2,
        },
        {
            name:  'website_domain',
            label:  'Current Domain Address (If Any)',
            type:  'text',
            placeholder:  "Enter Website's Current Domain",
            inputClass:  'input current-domain-input',
            description: 'Please write NA in response of this question if there is no current domain address',
            required:  true,
            step:  2,
        },
        {
            name:  'business_operations',
            label:  'Please describe your organization\'s operations in a few sentences.',
            type:  'text',
            placeholder:  "Enter Organization Operations",
            inputClass:  'input organization-steps-input',
            required:  true,
            step:  3,
        },
        {
            name:  'app_purpose',
            label:  'What will be the primary purpose of your mobile app?',
            type:  'radio',
            options:  [
                {value:  'relaying_business_information', text:  'Relaying Information About Your Business'},
                {value:  'buying_selling_online', text:  'Buying and(or) Selling Online'},
                {value:  'appointment_booking', text:  'Appointment Booking'},
                {value:  'office_productivity', text:  'Office Productivity'},
                {value:  'customer_service_improvement', text:  'Improve Your Customer Service'},
                {value:  'travel_apps', text:  'Travel Apps'},
                {value:  'medical_app', text:  'Medical App'},
            ],
            otherOption: true,
            required:  true,
            step:  3,
        },
        {
            name:  'app_look',
            label:  'What Overall Look and Feel & Flow would you like to portray with your Mobile App?',
            type:  'radio',
            options:  [
                {value:  'corporate', text:  'Corporate'},
                {value:  'fun', text:  'Fun'},
                {value:  'high_tech', text:  'High Tech'},
                {value:  'modern', text:  'Modern'},
                {value:  'classic', text:  'Classic'},
                {value:  'vintage', text:  'Vintage'},
            ],
            otherOption: true,
            required:  true,
            step:  3,
        },
        {
            name:  'app_specification',
            label:  'Please provide detail Information & functional specifications that your Mobile App will provide',
            type:  'text',
            placeholder:  "App Specifications",
            inputClass:  'input app_specifications-input',
            required:  true,
            step:  3,
        },
        {
            name:  'is_cms',
            label:  'Would you like to avail CMS features for your Mobile App?',
            description: "CMS is a system which lets you add content and make day-to-day updates on your mobile app.",
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
                {value:  'maybe', text:  'Maybe'},
            ],
            required:  true,
            step:  3,
        },
        {
            name:  'content_restriction',
            label:  'Is there any content restriction outside of the US?',
            type:  'text',
            placeholder:  "Restrictions",
            inputClass:  'input content-restriction-input',
            required:  true,
            step:  3,
        },
        {
            name:  'users_accomplishment',
            label:  'What goals do you have for users to accomplish in your Mobile App?',
            type:  'text',
            placeholder:  "User Accomplishment",
            inputClass:  'input users-accomplishment-input',
            required:  true,
            step:  3,
        },
        {
            name:  'app_design',
            label:  'Do you have wireframes or app designs (PSD/AI files) ready for the application? Or would you like to avail our services on App Design?',
            type:  'text',
            placeholder:  "App Design",
            inputClass:  'input app-design-input',
            required:  true,
            step:  3,
        },
        {
            name:  'publish_assistance',
            label:  'Would you like to avail assistance publish app on App Stores like Google play & Apple App Store?',
            type:  'text',
            placeholder:  "Enter your answer",
            inputClass:  'input app-design-input',
            required:  true,
            step:  3,
        },
        {
            name:  'app_pages_section',
            label:  'Pages/Section you require e.g. About Us, Services/Products, Contact Us',
            type:  'text',
            placeholder:  "Pages & Section",
            inputClass:  'input app-design-input',
            required:  true,
            step:  3,
        },
        {
            name:  'app_reference',
            label:  'Do you have any reference app(s) which works similar to your requirement? Please list the ones you found closest.',
            type:  'text',
            placeholder:  "App Reference",
            inputClass:  'input app-reference-input',
            description: 'This can be even your competitor apps.',
            required:  true,
            step:  3,
        },
        {
            name:  'app_users',
            label:  'Please provide details information about the users who will using your Mobile App?',
            type:  'text',
            placeholder:  "App Users",
            inputClass:  'input app-users-input',
            description: 'e.g. Admin, Manager, Customer, Buyer, Vendor',
            required:  true,
            step:  3,
        },
        {
            name:  'is_push_notification',
            label:  'Would you need push notifications for your application?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
                {value:  'maybe', text:  'Maybe'},
            ],
            required:  true,
            step:  3,
        },
        {
            name:  'is_third_party_api',
            label:  'Would you like to integrate or use any Third party API services for your application?',
            type:  'text',
            placeholder:  "Usage of Third Party APIs",
            inputClass:  'input is-third-party-api-input',
            description: 'e.g. Social network authentication such as Facebook, Google, Twitter, etc.',
            required:  true,
            step:  3,
        },
        {
            name:  'app_tagline',
            label:  'Do you have a tagline/slogan?',
            type:  'text',
            placeholder:  "App Slogan",
            inputClass:  'input app-slogan-input',
            required:  true,
            step:  4,
        },
        {
            name:  'is_app_logo',
            label:  'Do you have a high-resolution logo?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
                {value:  'maybe', text:  'Maybe'},
            ],
            required:  true,
            step:  4,
        },
        {
            name:  'is_logo_redesign',
            label:  'Have you considered redesigning your logo?',
            type:  'radio',
            options:  [
                {value:  'yes', text:  'Yes'},
                {value:  'no', text:  'No'},
                {value:  'maybe', text:  'Maybe'},
            ],
            required:  true,
            step:  4,
        },
        {
            name:  'app_font_style',
            label:  'Please specify if you would like to use a specific font style',
            type:  'text',
            placeholder:  "App Font Styles",
            inputClass:  'input app-font-style-input',
            required:  true,
            step:  4,
        },
        {
            name:  'app_colors',
            label:  'Are there any colors that you want incorporated into the Mobile App?',
            type:  'text',
            placeholder:  "App Colors",
            inputClass:  'input app-colors-input',
            required:  true,
            step:  4,
        },
        {
            name:  'comments',
            label:  'Anything else that you would like to add, feel free to use the space below',
            type:  'text',
            placeholder:  "Commets",
            inputClass:  'input comments-input',
            required:  true,
            step:  5,
        },
        {
            name: 'info',
            sectionType: 'heading',
            heading: 'All correspondence and documents provided will be treated as confidential between the client and the designer, unless consent has been granted by both parties involved',
            // description: '',
            step: 6,
        },
    ],
    formFieldsData: {
        email:  "",
        name:  "",
        phone:  "",
        business_name:  "",
        business_address:  "",
        business_phone:  "",
        business_email:  "",
        business_operations:  "",
        website_link:  "",
        website_hosting:  "",
        website_domain:  "",
        app_purpose:  "",
        app_tagline:  "",
        app_look:  "",
        app_design:  "",
        app_specification:  "",
        app_pages_section:  "",
        app_reference:  "",
        app_users:  "",
        app_font_style:  "",
        app_colors: "",
        is_cms:  "",
        is_push_notification:  "",
        is_app_logo:  "",
        is_logo_redesign:  "",
        is_third_party_api:  "",
        content_restriction:  "",
        users_accomplishment:  "",
        publish_assistance:  "",
        comments:  "",
    }

}
