<?php

namespace App\Services;

use App\Helper\Helper\StripeHelper;
use Carbon\Carbon;
use DateTime;
use GuzzleHttp\Client;
use Exception;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

class PaypalService
{
    private $baseUrl, $clientId, $clientSecret;

    function __construct($clientId, $clientSecret, $mode = 'testing')
    {
        $this->baseUrl = $mode === 'testing' ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
    }

    public function generateAccessToken()
    {
        try {
            $client = new Client();
            $response = $client->post("{$this->baseUrl}/v1/oauth2/token", [
                'auth' => [$this->clientId, $this->clientSecret],
                'form_params' => [
                    'grant_type' => 'client_credentials',
                ],
            ]);

            $data = json_decode($response->getBody(), true);
            return $data['access_token'];
        } catch (RequestException $e) {
            $error['success'] = false;
            if ($e->hasResponse()) {
                // If there is a response, extract the error details from PayPal
                $errorResponse = json_decode($e->getResponse()->getBody()->getContents(), true);
                $error['error'] = $errorResponse;
                // Example of logging or displaying the error message
                if (isset($errorResponse['name']) && isset($errorResponse['message'])) {
                    $errorName = $errorResponse['name']; // PayPal error name
                    $errorMessage = $errorResponse['message']; // PayPal error message
                    $debugInfo = $errorResponse['debug_id']; // PayPal debug info for tracing

                    $error['message'] = $errorResponse['message'];

                    Log::error("PayPal API Error: $errorName - $errorMessage, Debug ID: $debugInfo");
                } else {
                    Log::error("Unknown PayPal API Error");
                    $error['message'] = 'Unknown PayPal API Error';
                }
            } else {
                // Handle cases where no response is available
                Log::error("Request failed without a response: " . $e->getMessage());
                $error['message'] = $e->getMessage();
            }
            return $error;
        }
    }

    public function createOrder($customer, $card, $link, $type = 'paypal')
    {

        try {
            $accessToken = $this->generateAccessToken();
            $paypalRequestId = uniqid('paypal_', true);
            if ($type == 'paypal') {
                $payment_source['paypal']['name']['given_name'] = $customer['firstname'];
                $payment_source['paypal']['name']['surname'] = $customer['lastname'];
                $payment_source['paypal']['experience_context']['return_url'] = route('paypal.order.success', ["client_token" => $link['token'], 'paymentID' => $link['payment_id']]);
                $payment_source['paypal']['experience_context']['cancel_url'] = route('paypal.order.failed', ["client_token" => $link['token'], 'paymentID' => $link['payment_id']]);
            } else {
                $filtercard = str_replace(' ', '', $card['cardNo']);
                // Convert "05/28" to "2028/05"
                $dateTime = DateTime::createFromFormat('m/y', $card['card_date']);
                $formattedCardDate = $dateTime->format('Y-m');

                $payment_source['card']['name'] = $customer['firstname'] . ' ' . $customer['lastname'];
                $payment_source['card']['number'] = $filtercard;
                $payment_source['card']['expiry'] = $formattedCardDate;
                $payment_source['card']['security_code'] = $card['cvc'];
            }

            $client = new Client();
            $amount['currency_code'] = $link['currency'] ?? 'USD';
            $amount['value'] = $link['itemprice'];

            $response = $client->post("{$this->baseUrl}/v2/checkout/orders", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                    'PayPal-Request-Id' => $paypalRequestId,
                ],
                'json' => [
                    'intent' => 'CAPTURE',
                    'purchase_units' => [
                        [
                            'amount' => $amount,
                        ]
                    ],
                    'payment_source' => $payment_source,
                ],
            ]);

            return [
                "success" => true,
                "data" => json_decode($response->getBody(), true)
            ];
        } catch (RequestException $e) {
            $error['success'] = false;
            if ($e->hasResponse()) {
                // If there is a response, extract the error details from PayPal
                $errorResponse = json_decode($e->getResponse()->getBody()->getContents(), true);
                $error['error'] = $errorResponse;
                // Example of logging or displaying the error message
                if (isset($errorResponse['name']) && isset($errorResponse['message'])) {
                    $errorName = $errorResponse['name']; // PayPal error name
                    $errorMessage = $errorResponse['message']; // PayPal error message
                    $debugInfo = $errorResponse['debug_id']; // PayPal debug info for tracing

                    $error['message'] = $errorResponse['message'];

                    Log::error("PayPal API Error: $errorName - $errorMessage, Debug ID: $debugInfo");
                } else {
                    Log::error("Unknown PayPal API Error");
                    $error['message'] = 'Unknown PayPal API Error';
                }
            } else {
                // Handle cases where no response is available
                Log::error("Request failed without a response: " . $e->getMessage());
                $error['message'] = $e->getMessage();
            }
            return $error;
        }
    }

    public function captureOrder($id)
    {
        try {
            $accessToken = $this->generateAccessToken();
            $client = new Client();
            $response = $client->post("{$this->baseUrl}/v2/checkout/orders/$id/capture", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
            ]);

            return [
                "success" => true,
                "data" => json_decode($response->getBody(), true)
            ];
        } catch (RequestException $e) {
            $error['success'] = false;
            if ($e->hasResponse()) {
                // If there is a response, extract the error details from PayPal
                $errorResponse = json_decode($e->getResponse()->getBody()->getContents(), true);
                $error['error'] = $errorResponse;
                // Example of logging or displaying the error message
                if (isset($errorResponse['name']) && isset($errorResponse['message'])) {
                    $errorName = $errorResponse['name']; // PayPal error name
                    $errorMessage = $errorResponse['message']; // PayPal error message
                    $debugInfo = $errorResponse['debug_id']; // PayPal debug info for tracing

                    $error['message'] = $errorResponse['message'];

                    Log::error("PayPal API Error: $errorName - $errorMessage, Debug ID: $debugInfo");
                } else {
                    Log::error("Unknown PayPal API Error");
                    $error['message'] = 'Unknown PayPal API Error';
                }
            } else {
                // Handle cases where no response is available
                Log::error("Request failed without a response: " . $e->getMessage());
                $error['message'] = $e->getMessage();
            }
            return $error;
        }
    }

    public function GetOrder($id)
    {
        try {
            $accessToken = $this->generateAccessToken();
            $client = new Client();
            // Using GET request instead of POST
            $response = $client->get("{$this->baseUrl}/v2/checkout/orders/{$id}", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
            ]);

            $data = json_decode($response->getBody(), true);

            return [
                "success" => true,
                "data" => $data
            ];
        } catch (RequestException $e) {
            $error['success'] = false;
            if ($e->hasResponse()) {
                // If there is a response, extract the error details from PayPal
                $errorResponse = json_decode($e->getResponse()->getBody()->getContents(), true);
                $error['error'] = $errorResponse;
                // Example of logging or displaying the error message
                if (isset($errorResponse['name']) && isset($errorResponse['message'])) {
                    $errorName = $errorResponse['name']; // PayPal error name
                    $errorMessage = $errorResponse['message']; // PayPal error message
                    $debugInfo = $errorResponse['debug_id']; // PayPal debug info for tracing

                    $error['message'] = $errorResponse['message'];

                    Log::error("PayPal API Error: $errorName - $errorMessage, Debug ID: $debugInfo");
                } else {
                    Log::error("Unknown PayPal API Error");
                    $error['message'] = 'Unknown PayPal API Error';
                }
            } else {
                // Handle cases where no response is available
                Log::error("Request failed without a response: " . $e->getMessage());
                $error['message'] = $e->getMessage();
            }
            return $error;
        }
    }

    public function paymentMethod($formData, $link, $orderData, $type = 'paypal')
    {
        $pmID = StripeHelper::generateUniqueID('pm', 10);
        $billing_details['address']['city'] = $formData['city'];
        $billing_details['address']['country'] = $formData['country'];
        $billing_details['address']['postal_code'] = $formData['zipcode'];
        $billing_details['address']['state'] = $formData['statename'];
        $billing_details['email'] = $formData['clientemail'];
        $billing_details['name'] = $formData['firstname'] . ' ' . $formData['lastname'];
        $billing_details['phone'] = $formData['phonenum'];

        $metadata = [
            "token" => $formData['token'],
            "first_name" => $formData['first_name'],
            "last_name" => $formData['last_name'],
            "email" => $formData['email'],
            "phone" => $formData['phone'],
            "company" => $formData['company'],
            "address" => $formData['address'],
            "city" => $formData['city'],
            "state" => $formData['state'],
            "country" => $link->currencyCountry->aplha_code3,
            "brand_descriptor" => $formData['brand_descriptor']
        ];

        $paymentMethod = [
            "id" => $pmID,
            "object" => "payment_method",
            "billing_details" => $billing_details,
            "created" => Carbon::now()->timestamp,
            "metadata" => $metadata,
            "type" => "card",
            "data" => $orderData
        ];

        if ($type == 'paypal') {
            $paymentMethod['source'] = 'paypal';
            $paymentMethod['paypal'] = $orderData['links'];
        } else {
            $expiry = $orderData['payment_source']['card']['expiry'];

            if (!empty($expiry)) {
                $expiry = explode('-', $expiry);
            }

            $paymentMethod['source'] = 'card';
            $paymentMethod['card']['object'] = $orderData['payment_source']['card'];
            $paymentMethod['card']['brand'] = $orderData['payment_source']['card']['brand'];
            $paymentMethod['card']['country'] = $formData['country'];
            $paymentMethod['card']['display_brand'] = $orderData['payment_source']['card']['brand'];
            $paymentMethod['card']['exp_month'] = $expiry[1] ?? '';
            $paymentMethod['card']['exp_year'] = $expiry[0] ?? '';
            $paymentMethod['card']['last4'] = $orderData['payment_source']['card']['last_digits'];
        }

        $paymentMethod['status'] = $orderData['status'];

        return $paymentMethod;
    }

    public function paymentIntent($formData, $link, $orderData, $type = 'paypal')
    {
        $billing_details['address']['city'] = $formData['city'];
        $billing_details['address']['country'] = $formData['country'];
        $billing_details['address']['postal_code'] = $formData['zipcode'];
        $billing_details['address']['state'] = $formData['statename'];
        $billing_details['email'] = $formData['clientemail'];
        $billing_details['name'] = $formData['firstname'] . ' ' . $formData['lastname'];
        $billing_details['phone'] = $formData['phonenum'];

        $metadata = [
            "token" => $formData['token'],
            "first_name" => $formData['first_name'],
            "last_name" => $formData['last_name'],
            "email" => $formData['email'],
            "phone" => $formData['phone'],
            "company" => $formData['company'],
            "address" => $formData['address'],
            "city" => $formData['city'],
            "state" => $formData['state'],
            "country" => $link->currencyCountry->aplha_code3,
            "brand_descriptor" => $formData['brand_descriptor']
        ];

        $paymentIntent = [
            "id" => $orderData['id'],
            "object" => "payment_intent",
            "amount" => $link->price * 100,
            "created" => Carbon::now()->timestamp,
            "currency" => $link->currencyCountry->code,
            "description" => $link->item_name,
            "last_payment_error" => null,
            "metadata" => $metadata,
            "source" => $orderData,
            "statement_descriptor" => "Logo Web Servs LO",
            "status" => $orderData['status'],
        ];

        return $paymentIntent;
    }
}
