<?php

namespace App\Services;

use App\Models\PaymentLink;
use App\Models\User;
use App\Repositories\Customer\CustomerRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class CustomerService
{
    private $customer, $paymentLinkService, $customerRepository;

    function __construct(User $user, PaymentLinkService $paymentLinkService, CustomerRepository $customerRepository)
    {
        $this->customer = $user;
        $this->paymentLinkService = $paymentLinkService;
        $this->customerRepository = $customerRepository;
    }

    public function getCustomer($request, $user, $roles)
    {
        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Super Admin'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        $roles = $filteredRoles;

        $customersQuery = $this->customer::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'city', 'company', 'address', 'state', 'zipcode', 'country', 'stripe_customer_id', 'stripe_payment_method', 'created_at', 'target')
            ->role('customer')
            ->latest('id')
            ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', 'like', '%' . $request->phone . '%');
            })
            ->when(!empty($request->email), function ($query) use ($request) {
                $query->where('email', '=', $request->email);
            })
            ->when(!empty($request->card), function ($query) use ($request) {
                $query->where('stripe_payment_method', 'like', '%' . $request->card . '%');
            })
            ->when((!empty($request->date) && !empty($request->dateSearchBy) && !empty($request->durationType)), function ($query) use ($request) {
                $date = Carbon::today();
                if ($request->dateSearchBy == '=') {
                    if ($request->durationType == 'days') {
                        $date->modify(($request->dateSearchBy == '=' ? '-' : '-') . $request->date . ' days');
                    } elseif ($request->durationType == 'months') {
                        $date->modify(($request->dateSearchBy == '=' ? '-' : '-') . $request->date . ' months');
                    }
                    $query->whereDate('created_at', $date->format('Y-m-d H:i:s'));
                } else {
                    if ($request->durationType == 'days') {
                        $date->modify('-' . $request->date . ' days');
                    } elseif ($request->durationType == 'months') {
                        $date->modify('-' . $request->date . ' months');
                    }
                    $pointDate = $date->format('Y-m-d H:i:s');
                    $query->whereDate('created_at', $request->dateSearchBy, $pointDate);
                }
            });

        if ($user->hasRole('Super Admin')) {
            return $customersQuery->latest('id')->paginate(20)->withQueryString();
        } elseif ($user->hasAnyRole($roles)) {
            if ($user->can('Customer-Global')) {

                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');
              
                return $customersQuery
                 ->whereNotIn('created_by', $superAdminIds)
                ->latest('id')->paginate(20)->withQueryString();

            } else {
                return $customersQuery->whereHas('payments.link', function ($query) use ($user) {
                    $query->where('created_by', $user->id);
                })->latest('id')->paginate(20)->withQueryString();
            }

        } else {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
    }

    public function getCustomerDetails($id)
    {
        return $this->customer::where('id', '=', $id)->with([
            'invoices' => function ($invoice) {
                $invoice->with('payment');
            },
            'countryData:id,aplha_code3,country,symbol',
            'quotes',
            'paymentLogs' => function ($logs) {
                $logs->latest('created_at');
            },
            'payments' => function ($query) {
                $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                $query->with([
                    'link' => function ($linkQ) {
                        $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                        $linkQ->with('currencyCountry:id,aplha_code3,code,symbol', 'gateway:id,name,gateway,statement_descriptor,environment');
                    },
                    'paymentLogs' => function ($logs) {
                        $logs->latest('created_at');
                    },
                ]);
                $query->latest('id');
                $query->paginate(5);
            }
        ])->firstOrFail();
    }

    public function create($customerData, $paymentLinkID)
    {
        $customer = $this->customer::withTrashed()->select('id', 'email')->where("email", "=", $customerData->email);
        $paymentLink = $this->paymentLinkService->get($paymentLinkID);
        $customerExists = false;

        if ($customer->exists()) {
            $customerExists = true;
            $customerData = $customer->first();
            $customer_id = $customerData->id;
        } else {
            $customer = new User();
            $customer->first_name = $customerData->first_name;
            $customer->last_name = $customerData->last_name;
            $customer->email = $customerData->email;
            $customer->phone = $customerData->phone;
            $customer->company = $customerData->company;
            $customer->address = $customerData->address;
            $customer->city = $customerData->city;
            $customer->state = $customerData->state;
            $customer->zipcode = $customerData->zipcode;
            $customer->country = $customerData->country;
            $customer->stripe_customer_id = $customerData->stripe_customer_id ?? '';
            $customer->stripe = $customerData->stripe ?? '';
            $customer->created_by = $paymentLink['created_by'];
            $customer->created_at = Carbon::now();
            $customer->password = Hash::make('12345678');
            $customer->save();
            $customer_id = $customer->id;

            $roles = "Customer";
            $customer->assignRole($roles);
        }

        return [
            'id' => $customer_id,
            'existing_customer' => $customerExists
        ];
    }

    public function update($data, $customerID)
    {
        // $update = $this->customer::where('id', $customerID)->update($data);

        // if ($update == 0) {
        //     return false;
        // }

        // return true;


        $this->customerRepository->update($customerID, $data);
    }
}
