<?php

namespace App\Repositories\VideoBrief;

use App\Models\VideoBrief;
use App\Services\BriefLogService;
use App\Helper\BriefHelper;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Spatie\Permission\Models\Role;

class VideoBriefRepository implements VideoBriefInterface
{

    private $videoBrief, $briefLogService;

    public function __construct(VideoBrief $videoBrief, BriefLogService $briefLogService)
    {
        $this->videoBrief = $videoBrief;
        $this->briefLogService = $briefLogService;
    }

    public function getAllBriefs($request)
    {
        // $briefs = $this->videoBrief->select('id', 'encoded_id', 'name', 'email', 'phone', 'created_at');
        $briefs = $this->videoBrief->select('id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'industry', 'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        // Adjust time if provided
                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        // Adjust timezone
                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        // Adjust timezone
                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        // Apply date and time range filter
                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
            $query->whereHas('briefLink.customer', function ($q) use ($request) {
                $q->where('company', $request->company);
            });
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('video_brief', [
            'title' => 'Video Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'video.brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        // Retrieve roles that have the 'BriefForm-Global' permission
        $rolesWithBriefGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithBriefGlobalPermission)) {
            $briefs = $this->videoBrief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->videoBrief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $videoBrief = new $this->videoBrief;

            $videoBrief->brief_link_id = $data->brief_link_id;
            $videoBrief->email = $data->email;
            $videoBrief->name = $data->name;
            $videoBrief->phone = $data->phone;
            $videoBrief->project_name = $data->project_name;
            $videoBrief->industry = $data->industry;
            $videoBrief->website = $data->website;
            $videoBrief->social_media = $data->social_media;
            $videoBrief->business_description = $data->business_description;
            $videoBrief->business_services = $data->business_services;
            $videoBrief->audience = $data->audience;
            $videoBrief->competitor = $data->competitor;
            $videoBrief->u_s_p = $data->u_s_p;
            $videoBrief->comments = $data->comments;
            $videoBrief->video_purpose = $data->video_purpose;
            $videoBrief->video_duration = $data->video_duration;
            $videoBrief->reference = $data->reference;
            $videoBrief->style = $data->style;
            $videoBrief->color = $data->color;
            $videoBrief->voice = $this->arrayToString($data->voice);
            $videoBrief->artist_accent = $data->artist_accent;
            $videoBrief->animation_style = $data->animation_style;
            $videoBrief->reference_video = $data->reference_video;
            $videoBrief->deadline = $data->deadline;
            $videoBrief->additional_comments = $data->additional_comments;
            $videoBrief->customer_id = $data->customer_id ?? Auth::user()->id;

            $videoBrief->save();
            $videoBrief = $videoBrief->refresh();

            return [
                'id' => $videoBrief->id,
                'encoded_id' => $videoBrief->encoded_id,
                'brief_link_id' => $videoBrief->brief_link_id,
                'email' => $videoBrief->email ?? '',
                'name' => $videoBrief->name ?? '',
                'phone_number' => $videoBrief->phone ?? '',
                'industry' => $videoBrief->industry ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function getBrief($id){
        return $this->videoBrief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }

}
