<?php

namespace App\Repositories\MobileAppBrief;

use App\Models\Brief;
use App\Helper\BriefHelper;
use App\Models\MobileAppBrief;
use Spatie\Permission\Models\Role;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;

class MobileAppBriefRepository implements MobileAppBriefInterface
{

    private $brief;

    public function __construct(MobileAppBrief $brief)
    {
        $this->brief = $brief;
    }

    public function getAllBriefs($request)
    {
        $briefs = $this->brief->select(  'id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'business_name',  'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
                $query->where('business_name', $request->company);
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('mobile_app_brief', [
            'title' => 'Mobile App Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        $rolesWithMobleAppGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithMobleAppGlobalPermission)) {
            $briefs = $this->brief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->brief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $brief = new $this->brief;

            $brief->brief_link_id = $data->brief_link_id;
            $brief->customer_id = $data->customer_id ?? Auth::user()->id;
            $brief->device = $data->device ?? null;
            $brief->location = $data->location ?? null;
            $brief->email = $data->email;
            $brief->name = $data->name;
            $brief->phone = $data->phone ?? null;
            $brief->business_name = $data->business_name;
            $brief->business_address = $data->business_address;
            $brief->business_phone = $data->business_phone ?? null;
            $brief->business_email = $data->business_email ?? null;
            $brief->business_operations = $data->business_operations ?? null;
            $brief->website_link = $data->website_link ?? null;
            $brief->website_hosting = $data->website_hosting ?? null;
            $brief->website_domain = $data->website_domain ?? null;
            $brief->app_purpose = $data->app_purpose ?? null;
            $brief->app_tagline = $data->app_tagline ?? null;
            $brief->app_look = $data->app_look ?? null;
            $brief->app_design = $data->app_design ?? null;
            $brief->app_specification = $data->app_specification ?? null;
            $brief->app_pages_section = $data->app_pages_section ?? null;
            $brief->app_reference = $data->app_reference ?? null;
            $brief->app_users = $data->app_users ?? null;
            $brief->app_font_style = $data->app_font_style ?? null;
            $brief->app_colors = $data->app_colors ?? null;
            $brief->is_cms = $data->is_cms ?? null;
            $brief->is_push_notification = $data->is_push_notification ?? null;
            $brief->is_app_logo = $data->is_app_logo ?? null;
            $brief->is_logo_redesign = $data->is_logo_redesign ?? null;
            $brief->is_third_party_api = $data->is_third_party_api ?? null;
            $brief->content_restriction = $data->content_restriction ?? null;
            $brief->users_accomplishment = $data->users_accomplishment ?? null;
            $brief->publish_assistance = $data->publish_assistance ?? null;
            $brief->comments = $data->comments ?? null;

            $brief->save();
            $brief = $brief->refresh();

            return [
                'id' => $brief->id,
                'encoded_id' => $brief->encoded_id,
                'brief_link_id' => $brief->brief_link_id,
                'email' => $brief->email ?? '',
                'name' => $brief->name ?? '',
                'phone' => $brief->phone ?? '',
                'business_name' => $brief->business_name ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function getBrief($id){
        return $this->brief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }
}
