<?php

namespace App\Repositories\Logs;

interface LogInterface
{
    /**
     * Add a new log entry.
     *
     * @param array $data An associative array containing log data. Expected keys include 'class_id', 'class_type', 'activity', 'type', 'request', 'response', 'created_by', 'event', 'pin'.
     * @return array The updated log data.
     * @throws \Exception If there is an error while logging.
     */
    public function log(array $data);

    /**
     * Update an existing log entry by its ID.
     *
     * @param string $logId The ID of the log entry to update.
     * @param string $classID The class ID associated with the log.
     * @param array $data An associative array with the updated log data.
     * @return array The updated log entry.
     * @throws \Exception If there is an error while updating or if the log entry is not found.
     */
    public function updateLog(string $logId, string $classID, array $data);

    /**
     * Pin or unpin a log entry by its ID.
     *
     * @param string $logId The ID of the log entry to pin or unpin.
     * @param string $classID The class ID associated with the log.
     * @param int $pin The pin status (1 to pin, 0 to unpin).
     * @return array The updated log entry with the pin status.
     * @throws \Exception If there is an error while pinning/unpinning or if the log entry is not found.
     */
    public function pinUnpinLog(string $logId, string $classID, int $pin);

    /**
     * Retrieve a specific log entry by its ID.
     *
     * @param string $logId The ID of the log entry to retrieve.
     * @param string $classID The class ID associated with the log.
     * @return array The log entry data.
     * @throws \Exception If there is an error while retrieving or if the log entry is not found.
     */
    public function getLog(string $logId, string $classID);

    /**
     * Delete a specific log entry by its ID.
     *
     * @param string $logId The ID of the log entry to delete.
     * @param string $classID The class ID associated with the log.
     * @return bool True if the log entry was deleted, false otherwise.
     * @throws \Exception If there is an error while deleting or if the log entry is not found.
     */
    public function deleteLog(string $logId, string $classID);

    /**
     * Retrieve all log entries for a specific class ID.
     *
     * @param string $classID The class ID associated with the logs.
     * @return array An array of log entries.
     * @throws \Exception If there is an error while retrieving the logs.
     */
    public function getAllLogs(string $classID);
}
