<?php

namespace App\Repositories\Coupon;

use App\Helper\Helper\StripeHelper;
use App\Models\Coupon;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;

class CouponRepository implements CouponInterface
{
    private $coupon;

    /**
     * Constructor to inject dependencies.
     *
     * @param Coupon $coupon
     * @param Carbon $carbon
     */
    public function __construct(Coupon $coupon)
    {
        $this->coupon = $coupon;
    }

    /**
     * Retrieve a paginated list of coupons based on request parameters.
     *
     * @param Request $request
     * @return mixed
     */
    public function getCoupons(Request $request)
    {
        $query = $this->coupon::query()
            ->select('id', 'name', 'description', 'coupon_id', 'discount', 'discount_type', 'duration', 'date_from', 'date_to', 'currency', 'created_by', 'created_at')
            ->with('currencyCountry')
            ->latest('id');

        if ($request->has('status_all')) {
            $statusAll = $request->status_all;
            if ($statusAll === 'active') {
                $query->where(function ($q) {
                    $q->whereIn('duration', ['forever', 'once'])
                      ->orWhere(function ($q) {
                          $q->whereDate('date_from', '<=', now())
                            ->whereDate('date_to', '>=', now());
                      });
                });
            } elseif ($statusAll === 'deactive') {
                $query->where(function ($q) {
                    $q->whereDate('date_from', '>', now())
                      ->orWhereDate('date_to', '<', now());
                });
            }
        }

        $coupons = $query->paginate(20)->withQueryString();

        $currentDate = Carbon::now();
        foreach ($coupons as $coupon) {
            if ($coupon->date_from && $coupon->date_to) {
                $isValid = $currentDate->between(Carbon::parse($coupon->date_from), Carbon::parse($coupon->date_to));
            } else {
                $isValid = in_array($coupon->duration, ['forever', 'once']);
            }
            $coupon->valid = $isValid;
        }

        return $coupons;
    }

    /**
     * Retrieve detailed information about a specific coupon.
     *
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function getCouponDetail($id)
    {
        try {
            $coupon = $this->coupon
                ->select('id', 'name', 'description', 'coupon_id', 'discount', 'data', 'discount_type', 'duration', 'date_from', 'date_to', 'currency', 'created_by', 'created_at')
                ->with('couponLogs', 'currencyCountry')
                ->findOrFail($id);

            if ($coupon->date_from && $coupon->date_to) {
                $isValid = Carbon::now()->between(Carbon::parse($coupon->date_from), Carbon::parse($coupon->date_to));
            } else {
                $isValid = in_array($coupon->duration, ['forever', 'once']);
            }

            $coupon->valid = $isValid;
            return $coupon;
        } catch (ModelNotFoundException $e) {
            throw new ModelNotFoundException('Coupon not found.');
        } catch (Exception $ex) {
            throw new Exception('Failed to fetch coupon details: ' . $ex->getMessage());
        }
    }

    /**
     * Create a new coupon record in the database.
     *
     * @param Request $request
     * @return mixed
     * @throws Exception
     */
    public function create(Request $request)
    {
        try {
            $coupon = new Coupon();

            $coupon->name = str_replace(' ', '', $request->input('name'));

            $coupon->description = $request->input('description');
            $coupon->discount = $request->input('discount');
            $coupon->coupon_id = $request->filled('coupon_id')
                ? $request->input('coupon_id')
                : StripeHelper::generateUniqueID('cou', 5);
            $coupon->discount_type = $request->input('type');
            $coupon->currency = $request->input('currency');
            $coupon->duration = $request->input('duration');
            $coupon->created_by = $request->user()->id;

            if ($coupon->duration === "multiple_months") {
                $coupon->date_from = Carbon::today();
                $coupon->date_to = Carbon::today()->addMonths($request->input('duration_months'));
            }

            $coupon->save();

            return $coupon;
        } catch (Exception $ex) {
            throw new Exception('Failed to create coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Update an existing coupon record in the database.
     *
     * @param Request $request
     * @return mixed
     * @throws Exception
     */
    public function update(Request $request)
    {
        try {
            $coupon = $this->coupon::find($request->id);

            if (!$coupon) {
                throw new ModelNotFoundException('Coupon not found.');
            }

            if ($request->filled('name')) {
                $coupon->name = str_replace(' ', '', $request->input('name'));
            }

            if ($request->filled('description')) {
                $coupon->description = $request->input('description');
            }

            if ($request->filled('discount')) {
                $coupon->discount = $request->input('discount');
            }

            if ($request->filled('type')) {
                $coupon->discount_type = $request->input('type');
            }

            if ($request->filled('currency')) {
                $coupon->currency = $request->input('currency');
            }

            if ($request->filled('coupon_id')) {
                $coupon->coupon_id = $request->input('coupon_id');
            }

            if ($request->filled('duration')) {
                $coupon->duration = $request->input('duration');

                if ($coupon->duration === "multiple_months") {
                    $coupon->date_from = Carbon::today();
                    $coupon->date_to = Carbon::today()->addMonths($request->input('duration_months'));
                } else {
                    $coupon->date_from = null;
                    $coupon->date_to = null;
                }
            }

            $coupon->updated_by = $request->user()->id;
            $coupon->save();

            return $coupon;
        } catch (ModelNotFoundException $e) {
            throw new ModelNotFoundException('Coupon not found.');
        } catch (Exception $ex) {
            throw new Exception('Failed to update coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Delete a coupon by its ID.
     *
     * @param int $id
     * @return array
     * @throws Exception
     */
    public function deleteById($id)
    {
        try {
            $coupon = $this->coupon::find($id);
            if (!$coupon) {
                throw new ModelNotFoundException('Coupon not found.');
            }
            $coupon->delete();
            return ['message' => 'Coupon deleted successfully'];
        } catch (ModelNotFoundException $e) {
            throw new ModelNotFoundException('Coupon not found.');
        } catch (Exception $ex) {
            throw new Exception('Failed to delete coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Retrieve a coupon by its ID.
     *
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function getCouponById($id)
    {
        try {
            return $this->coupon->with('currencyCountry', 'couponLogs')->findOrFail($id);
        } catch (ModelNotFoundException $e) {
            throw new ModelNotFoundException('Coupon not found.');
        } catch (Exception $ex) {
            throw new Exception('Failed to fetch coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Retrieve a coupon by its name.
     *
     * @param string $name
     * @return mixed
     * @throws Exception
     */
    public function getCouponByName($name)
    {
        try {
            return $this->coupon->where('name', $name)->firstOrFail();
        } catch (ModelNotFoundException $e) {
            throw new ModelNotFoundException('Coupon not found.');
        } catch (Exception $ex) {
            throw new Exception('Failed to fetch coupon by name: ' . $ex->getMessage());
        }
    }
}
