<?php

namespace App\Repositories\Brief;

use App\Models\Brief;
use App\Helper\BriefHelper;
use App\Services\BriefLogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\Services\LogsService;
use Spatie\Permission\Models\Role;

class BriefRepository implements BriefInterface
{

    private $brief, $briefLogService, $logsService;

    public function __construct(Brief $brief, BriefLogService $briefLogService)
    {
        $this->brief = $brief;
        $this->briefLogService = $briefLogService;
        $this->logsService = new LogsService('/logs/briefLogs');
    }

    public function getAllBriefs($request)
    {
        $briefs = $this->brief->select('id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'industry', 'business_name', 'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        // Adjust time if provided
                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        // Adjust timezone
                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        // Adjust timezone
                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        // Apply date and time range filter
                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
                $query->where('business_name', $request->company);
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('logo_brief', [
            'title' => 'Logo Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getBrief($id){
        return $this->brief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        // Retrieve roles that have the 'BriefForm-Global' permission
        $rolesWithBriefGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithBriefGlobalPermission)) {
            $briefs = $this->brief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->brief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $brief = new $this->brief;

            $brief->brief_link_id = $data->brief_link_id;
            $brief->email = $data->email;
            $brief->name = $data->name;
            $brief->designation = $data->designation;
            $brief->phone = $data->phone;
            $brief->business_name = $data->business_name;
            $brief->slogan = $data->slogan;
            $brief->industry = $data->industry;
            $brief->business_description = $data->business_description;
            $brief->services = $data->services;
            $brief->audience = $data->audience;
            $brief->website = $data->website;
            $brief->competitor = $data->competitor;
            $brief->logo_title = $this->arrayToString($data->logo_title);
            $brief->logo_idea = $data->logo_idea;
            $brief->logo_message = $data->logo_message;
            $brief->logo_graphical_element = $data->logo_graphical_element;
            $brief->logo_customer_likes = $data->logo_customer_likes;
            $brief->logo_customer_dont_likes = $data->logo_customer_dont_likes;
            $brief->font_base_logo = $this->arrayToString($data->font_base_logo);
            $brief->graphic_base_logo = $this->arrayToString($data->graphic_base_logo);
            $brief->logo_color = $data->logo_color;
            $brief->logo_avoid_color = $data->logo_avoid_color;
            $brief->logo_font = $data->logo_font;
            $brief->logo_look = $data->logo_look;
            $brief->logo_use = $this->arrayToString($data->logo_use);
            $brief->business_growth = $this->arrayToString($data->business_growth ?? []);
            $brief->logo_budget = $data->logo_budget;
            $brief->contact_time = $data->contact_time;
            $brief->anything_else = $data->anything_else;
            $brief->completion_date = $data->completion_date;
            $brief->customer_id = $data->customer_id ?? Auth::user()->id;

            $brief->save();
            $brief = $brief->refresh();

            // $this->briefLogService->log([
            //     'brief_type' => 'logo_brief',
            //     'brief_id' => $brief->id,
            //     'activity' => 'Logo Brief has been created by ' . $brief->email . '.',
            //     'type' => 'logo_brief.create',
            //     'created_by' => Auth::user()->id,
            // ]);

            // $data = ['class_id' => $brief->id, 'class_type' => get_class($brief), 'activity' => 'Logo Brief has been created by ' . $brief->email . '.', 'type' => 'create', 'request' => '', 'response' => '', 'created_by' => Auth::user()->id, 'event' => null, 'pin' => 0];
            // $this->logsService->log($data);

            return [
                'id' => $brief->id,
                'encoded_id' => $brief->encoded_id,
                'brief_link_id' => $brief->brief_link_id,
                'email' => $brief->email ?? '',
                'name' => $brief->name ?? '',
                'phone_number' => $brief->phone ?? '',
                'business_name' => $brief->business_name ?? '',
                'industry' => $brief->industry ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }
}
