<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class CustomerRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {

        $customerID = $this->route('clientID') ?? null;

        return [
            'id' => 'nullable|integer',
            'first_name' => 'nullable|required_without_all:id,metadata',
            'last_name' => 'nullable|required_without_all:id,metadata',
            'email' => 'nullable|required_without_all:id,metadata|email|unique:users,email,' . $customerID,
            'phone' => 'nullable|required_without_all:id,metadata|min:8|unique:users,phone,' . $customerID,
            'metadata' => 'nullable',
            'city' => 'nullable',
            'company' => 'nullable',
            'address' => 'nullable',
            'state' => 'nullable',
            'zipcode' => 'nullable',
            'country' => 'nullable',
            'image' => 'nullable',
            'stripe_customer_id' => 'nullable',
            'stripe' => 'nullable',
            'stripe_pm_id' => 'nullable',
            'stripe_payment_method' => 'nullable',
            'dashboard_data' => 'nullable',
        ];
    }


    /**
     * Get the error messages for the defined validation rules.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'first_name.required_without' => 'The first name is required.',
            'last_name.required_without' => 'The last name is required.',
            'email.required_without' => 'The email is required.',
            'email.email' => 'The email must be a valid email address.',
            'email.unique' => 'The email address is already in use by another user.',
            'phone.required_without' => 'The phone number is required.',
            'phone.min' => 'The phone number must be at least 8 characters.',
            'phone.unique' => 'The phone number is already in use by another user.',
        ];
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(response()->json(['errors' => $validator->errors()], 200));
    }
}
