<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Support\Facades\Validator;
use App\Models\PaymentLink;
use App\Models\CountryCurrencies;
use Illuminate\Support\Facades\Session;
use App\Services\LogService;
use App\Services\BinApiService;
use App\Services\IpGeoLocationService;
use App\Helper\Helper\StripeHelper;
use DateTime;
use Illuminate\Support\Facades\Http;
use App\Helper\Helper\NotificationHelper;
use App\Helper\PaymentHelper;
use App\Http\Requests\PaymentRequest;
use App\Jobs\ApiLogJob;
use App\Models\Coupon;
use App\Services\Payment\Paypal\PaymentService as PaypalPaymentService;
use App\Services\PaypalService;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;
use App\Exceptions\StripeException;
use GuzzleHttp\Exception\RequestException;


class PayPalController extends Controller
{
    private $logRepository;
    private $neutrinoApi;
    private $geoLocation;

    public function __construct(LogService $log, BinApiService $neutrinoApi, IpGeoLocationService $geoLocation)
    {
        $this->logRepository = $log;
        $this->neutrinoApi = $neutrinoApi;
        $this->geoLocation = $geoLocation;
    }
    // order creation for paypal both modes
    public function createOrder(PaymentRequest $request, PaypalPaymentService $paymentService)
    {
        $paymentID = '';
        $token = '';
        $type = '';
        $orderToken = '';

        try {
            $formData = (array) $request->all();
            $paymentDetails = (object) PaymentHelper::itemArray($formData);
            $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            if ($paymentDetails->coupon_id > 0) {
                $coupon = Coupon::select('id', 'discount', 'discount_type')->where('id', $paymentDetails->coupon_id)->first();
                $discount = PaymentHelper::claimCoupon($item_detail->price, $coupon->discount, $coupon->discount_type);

                $item_detail->coupon_id = $coupon->id;
                $item_detail->price = $discount['amount'];
                $item_detail->discount = $discount['discount'];
                $item_detail->discount_type = $discount['discountType'];
                $item_detail->save();

                $item_detail->refresh();
            }
            $paypalService = app()->make(PaypalService::class, [
                'clientId' => $item_detail->gateway->public_key,
                'clientSecret' => $item_detail->gateway->secret_key,
                'mode' => $item_detail->gateway->environment
            ]);

            $orderCustomer['firstname'] = $formData['firstname'];
            $orderCustomer['lastname'] = $formData['lastname'];

            $card['cardNo'] = $formData['cardNo'];
            $card['card_date'] = $formData['card_date'];
            $card['cvc'] = $formData['cvc'];

            $link['token'] = $formData['token'];
            $link['itemprice'] = $item_detail->price;
            $link['currency'] = $item_detail->currencyCountry->code;

            $paymentDetails->currency_country = $item_detail->currencyCountry;
            $paymentDetails->link_id = $item_detail->id;

            $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
            $location = $this->geoLocation->ipLocation($request->ip());
            $geoLocate = $this->neutrinoApi->geolocate(address: $paymentDetails->address, houseNumber: "", street: "", city: $paymentDetails->city, county: $paymentDetails->address, state: $paymentDetails->state, postalCode: $paymentDetails->zipcode, countryCode: "", languageCode: 'en', fuzzySearch: false);
            $geoAddress['geolocation'] = !empty($geoLocate['locations']) ? $geoLocate['locations'] : '';
            $locationResponse = !empty($location->body()) ? array_merge(json_decode($location->body(), true), $geoAddress) : $geoAddress;

            $updatePayment = [
                'location' => response()->json($locationResponse)->content(),
                'device' => response()->json($device)->content()
            ];

            if ($request->card_type == 'creditCard') {
                $card8Digits = substr($paymentDetails->cardNo, 0, 8);
                $paymentDetails->cardNo = StripeHelper::maskNumber($paymentDetails->cardNo);

                StripeHelper::removeKey($paymentDetails, 'card_number');
                StripeHelper::removeKey($paymentDetails, 'card_cvc');
                StripeHelper::removeKey($paymentDetails, 'card_date');

                $binLookup = $this->neutrinoApi->binLookup($card8Digits, $request->ip());
                $updatePayment['bin'] = response()->json($binLookup)->content();
            }

            // Creating Customer
            $customer = $paymentService->addCustomer($paymentDetails, $item_detail);

            // Begin Payment Transaction
            $payment = $paymentService->initiate($customer['internal'], $item_detail, $request->ip(), $item_detail->comment, "");

            $paymentService->generateFile($item_detail->token, $paymentDetails, $payment->id, $item_detail->created_by);

            $delay = Carbon::now()->addMinutes(2);
            $this->queueApiLog('NEUTRINO UA-Lookup', ['user-agent' => $request->header('User-Agent')], $updatePayment['device'], $payment->id, $item_detail->created_by, $delay);
            $this->queueApiLog('NEUTRINO GeoAddress', ['ip' => $request->ip()], $updatePayment['location'], $payment->id, $item_detail->created_by, $delay);

            if ($request->card_type == 'creditCard') {
                $this->queueApiLog('NEUTRINO Bin-Lookup', ['bin' => $card8Digits], $updatePayment['bin'], $payment->id, $item_detail->created_by, $delay);
            }

            $paymentResponse['success'] = true;
            $paymentResponse['data'] = array(
                'payment_id' => $payment->id,
                'customer_id' => $customer['internal'],
            );

            $payment->update($updatePayment);
            $payment->save();

            $link['payment_id'] = $payment->id;

            $type = $request->card_type;
            $paymentID = $link['payment_id'];
            $token = $item_detail->token;

            if ($request->card_type == 'creditCard') {
                $order = $paypalService->createOrder($orderCustomer, $card, $link, $type = 'creditCard');
            } else {
                $order = $paypalService->createOrder($orderCustomer, $card, $link, $type = 'paypal');

            }

            $captureStatus = !empty($order['data']['purchase_units'][0]['payments']['captures'][0]['status']) ? $order['data']['purchase_units'][0]['payments']['captures'][0]['status'] : false;

            if ($captureStatus) {
                $orderToken = $order['data']['id'];
                if ($captureStatus == 'COMPLETED' || $captureStatus == 'PENDING') {
                    $route = route('paypal.order.success', [
                        'client_token' => $item_detail->token,
                        'type' => $request->card_type ?? 'paypal',
                        'paymentID' => $payment->id,
                        'token' => $order['data']['id']
                    ]);

                    return response()->json([
                        'success' => true,
                        'data' => [
                            'id' => $order['data']['id'],
                            'links' => !empty($order['data']['links']) ? $order['data']['links'] : [],
                            'payment_id' => $payment->id,
                            'token' => $item_detail->id,
                            'type' => $request->card_type ?? 'paypal',
                            'route' => $route
                        ]
                    ]);
                } else if ($captureStatus == 'DECLINED') {
                    $processorResponse = $order['data']['purchase_units'][0]['payments']['captures'][0]['processor_response'] ?? [];
                    $responseCode = $processorResponse['response_code'] ?? '';
                    throw new Exception($responseCode);
                }
            }

            if ($order['success']) {
                $route = route('paypal.order.success', [
                    'client_token' => $item_detail->token,
                    'type' => $request->card_type ?? 'paypal',
                    'paymentID' => $payment->id,
                    'token' => $order['data']['id']
                ]);

                return response()->json([
                    'success' => true,
                    'data' => [
                        'id' => $order['data']['id'],
                        'links' => !empty($order['data']['links']) ? $order['data']['links'] : [],
                        'payment_id' => $payment->id,
                        'token' => $item_detail->id,
                        'type' => $request->card_type ?? 'paypal',
                        'route' => $route
                    ]
                ]);
            } else {
                throw new Exception('Order not created yet please try again.');
            }
        } catch (\Exception $ex) {
            return response()->json([
                'success' => false,
                'data' => [
                    'route' => route('paypal.order.failed', [
                        'client_token' => $token,
                        'paymentID' => $paymentID,
                        'type' => $type,
                        'token' => $orderToken
                    ]),
                    'client_token' => $token,
                    'paymentID' => $paymentID,
                    'type' => $type,
                    'token' => $orderToken
                ],
                'message' => $ex->getMessage()
            ], 500); // Use a 500 status code for server errors
        }
    }

    public function onSuccess(Request $request, $client_token, $paymentID, $type = 'paypal', PaypalPaymentService $paymentService)
    {
        try {
            $payment = Payment::where('id', $paymentID)
                ->with('customer')->first();

            $item_detail = PaymentLink::where("token", "=", $client_token)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            $paypalService = app()->make(PaypalService::class, [
                'clientId' => $item_detail->gateway->public_key,
                'clientSecret' => $item_detail->gateway->secret_key,
                'mode' => $item_detail->gateway->environment
            ]);

            if ($type == 'paypal') {
                $capture = $paypalService->captureOrder($request->token);
            } else {
                $capture = $paypalService->GetOrder($request->token);
            }

            if ($capture['success']) {
                $paymentOrder = $paymentService->process($payment, $payment->customer, $item_detail, $capture['data'], $type);
                $payment->intent = $paymentOrder['payment_intent'];
                $payment->paymentMethod = $paymentOrder['payment_method'];
                $payment->intent_id = $capture['data']['id'];
                $payment->charge_id = $capture['data']['purchase_units'][0]['payments']['captures'][0]['id'];
                $payment->status = 1;
                $payment->save();

                InvoiceController::create($payment->id, false, 1);


                $success_data = array(
                    "payment_gateway" => $item_detail->payment_gateway,
                    "token" => $item_detail->token,
                    "original_price" => $item_detail->original_price,
                    "item_name" => $item_detail->item_name,
                    "price" => $item_detail->price,
                    "currency_country" => $item_detail->currencyCountry,
                    "payment_id" => $payment->id,
                    "discount" => (!empty($payment->discount) ? $payment->discount : 0),
                );

                if (!empty($item_detail->comment)) {
                    $commentLog = array(
                        'activity' => $item_detail->comment,
                        'type' => 'note',
                        'request' => '',
                        'response' => '',
                        'code' => 200,
                        'loggable_id' => $payment->id,
                        'created_by' => $item_detail->created_by,
                    );

                    $this->logRepository->log('payment', $commentLog);
                }

                $this->logRepository->log('payment', [
                    'activity' => 'Payment succeeded',
                    'type' => 'success',
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($paymentOrder['payment_intent'])->content(),
                    'code' => 200,
                    'loggable_id' => $payment->id,
                    'created_by' => $item_detail->created_by,
                ]);

                // User::where('id', $payment->customer_id)->update([
                //     'stripe_pm_id' => $paymentMethod->id,
                //     'stripe_payment_method' => response()->json($paymentMethod)->content()
                // ]);

                NotificationHelper::notify('payment', 'Payment-Received: Amount of ' . $item_detail->currencyCountry->symbol . $payment->price . ' ' . 'has been received', [
                    'id' => $payment->id,
                    'price' => $payment->price,
                    'message' => 'Payment successful',
                ], $item_detail->created_by);

                Session::flash("success_data", $success_data);
                $item_detail->status = 2;
                $item_detail->save();

                return Redirect::route('paynow.success', ["token" => $item_detail->token]);
            } else {
                throw new Exception('Payment not successfull.');
            }
        } catch (Exception $ex) {
            dd($ex->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => $ex->getMessage()
            ]);
        }
    }

    public function onFailed(Request $request, $client_token, $paymentID)
    {

        try {
            $payment = Payment::find($paymentID);
            $item_detail = PaymentLink::where("token", "=", $client_token)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            $paypalService = app()->make(PaypalService::class, [
                'clientId' => $item_detail->gateway->public_key,
                'clientSecret' => $item_detail->gateway->secret_key,
                'mode' => $item_detail->gateway->environment
            ]);

            $order = $paypalService->GetOrder($request->token);
            $errorMessage = [];
            // dd($order);
            if ($order['success']) {
                if (!empty($order['data']['status'])) {
                    // dd($order);
                    $orderStatus = $order['data']['purchase_units'][0]['payments']['captures'][0]['status'] ?? '';
                    if ($order['data']['status'] == 'PAYER_ACTION_REQUIRED') {
                        $errorMessage['code'] = "PAYER_ACTION_REQUIRED";
                        $errorMessage['title'] = "The payment can't be authorized.";
                        $errorMessage['message'] = "The payment requires authentication to proceed. If your are off session, you need to return to your application and complete the payment.";
                        $errorMessage['chat_code'] = "";
                    } else {
                        $processorResponse = $order['data']['purchase_units'][0]['payments']['captures'][0]['processor_response'] ?? [];
                        $responseCode = $processorResponse['response_code'] ?? '';
                        $geterrorMessage = getPaypalErrorMessage($responseCode);
                        $errorMessage['code'] = $geterrorMessage['code'];
                        $errorMessage['title'] = $geterrorMessage['title'];
                        $errorMessage['message'] = $geterrorMessage['message'];
                        $errorMessage['chat_code'] = $geterrorMessage['chat_code'];
                    }

                    $this->logRepository->log('payment', [
                        'activity' => 'Payment Failed because of ' . $orderStatus,
                        'type' => 'failed',
                        'request' => response()->json($request->all())->content(),
                        'response' => response()->json($errorMessage)->content(),
                        'code' => 402,
                        'loggable_id' => $payment->id,
                        'created_by' => $item_detail->created_by,
                    ]);

                    if ($payment->status == 3) {
                        $payment->status = 2;
                        $payment->save();
                    }

                    $payment->gateway_message = response()->json($errorMessage)->content();
                    $payment->save();
                }
            }

            $chatAgentMessage = (!empty($errorMessage["chat_code"]) ? $errorMessage["chat_code"] : $errorMessage["message"]);
            $sessional_time = Carbon::now()->addMinutes(10)->timestamp;

            $failed_data = array(
                "token" => $item_detail->token,
                'current_time' => Carbon::now()->timestamp,
                'session' => $sessional_time,
                'currency_country' => $item_detail->currencyCountry,
                'price' => $item_detail->price,
                'item_name' => $item_detail->item_name,
                'title' => 'Payment Failed | ' . $item_detail->item_name . ' (' . $item_detail->currencyCountry->code . $item_detail->price . ') | Message: ' . $chatAgentMessage,
                'heading' => 'Payment Failed',
                'message' => [
                    'title' => $errorMessage['title'],
                    'code' => $errorMessage["code"],
                    'chat_agent' => $chatAgentMessage,
                    'customer' => $errorMessage["message"]
                ]
            );

            NotificationHelper::notify('payment-failed', 'Payment-Failed: Amount of ' . $item_detail->currencyCountry->code . $payment->price . ' ' . 'has been failed', [
                'id' => $payment->id,
                'price' => $payment->price,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($errorMessage)->content(),
                'code' => 402
            ], $item_detail->created_by);

            Session::flash("failed_data", $failed_data);
            Session::reflash();
            if (Session()->get('payment_gateway')['gateway'] == 'paypal') {
                return Redirect::route('paynow.failed', ["token" => $item_detail->token]);
            }
            return response()->json(['success' => false, 'route' => route('paynow.failed', ["token" => $item_detail->token])], 200);

        } catch (Exception $ex) {
            return response()->json([
                'success' => false,
                'message' => $ex->getMessage()
            ]);
        }
    }

    public function queueApiLog($apiName, $request, $response, $paymentID, $createdBy, $delay)
    {
        dispatch(new ApiLogJob("A request to access ($apiName) api is initiated", 'api.call', response()->json($request)->content(), response()->json([])->content(), 200, $paymentID, $createdBy, $event = 1))->delay($delay);
        dispatch(new ApiLogJob('waiting_for_response', 'api.call.status', response()->json([])->content(), response()->json([])->content(), 200, $paymentID, $createdBy, $event = 1))->delay($delay);
        dispatch(new ApiLogJob("A request to retrieve data from ($apiName) API is completed", 'api.call', response()->json($request)->content(), $response, 200, $paymentID, $createdBy, $event = 0))->delay($delay);
    }
}
