<?php

namespace App\Http\Controllers\Admin;

use App\Exports\UsersExport;
use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Imports\CustomerImport;
use App\Models\CountryCurrencies;
use App\Models\User;
use App\Repositories\Customer\CustomerInterface;
use App\Services\Export\CustomerExport;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Maatwebsite\Excel\Facades\Excel;
use Spatie\Permission\Models\Role;
use App\Helper\ActivityLogHelper;
use App\Helper\BriefHelper;
use Illuminate\Validation\Rule;

class CustomerController extends Controller
{
    private $customer, $customerExport;

    function __construct(CustomerInterface $customer, CustomerExport $customerExport)
    {
        $this->customer = $customer;
        $this->customerExport = $customerExport;
    }

    public function list(Request $request)
    {
        $user = $request->user();
        $page["title"] = "Customers - PaymentModule";

        $columns = array(
            array("key" => "name", "component" => 'NameColumn', "text" => "Name", "visible" => true, "fixed" => true),
            array("key" => "email", "component" => 'EmailColumn', "text" => "Email", "visible" => true),
            array("key" => "payment_method", "component" => 'PaymentMethodColumn', "text" => "Default payment method", "visible" => true),
            array("key" => "created_at", "component" => 'CreatedAtColumn', "text" => "Created", "visible" => true),
            array("key" => "card_holder", "component" => 'CardHolderColumn', "text" => "Cardholder Name", "visible" => true),
            array("key" => "customer", "component" => 'CustomerColumn', "text" => "Customer", "visible" => true),
            array("key" => "phone", "text" => "Phone", "visible" => false),
            array("key" => "city", "text" => "City", "visible" => false),
            array("key" => "company", "text" => "Company", "visible" => false),
            array("key" => "address", "text" => "Address", "visible" => false),
            array("key" => "state", "text" => "State", "visible" => false),
            array("key" => "zipcode", "text" => "Zip", "visible" => false),
            array("key" => "country", "text" => "Country", "visible" => false),
        );

        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Super Admin'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        $roles = $filteredRoles;

        $customersQuery = User::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'city', 'company', 'address', 'state', 'zipcode', 'country', 'stripe_customer_id', 'stripe_payment_method', 'created_at', 'target')
            ->role('customer')
            ->latest('id')
            ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', 'like', '%' . $request->phone . '%');
            })
            ->when(!empty($request->email), function ($query) use ($request) {
                $query->where('email', '=', $request->email);
            })
            ->when(!empty($request->card), function ($query) use ($request) {
                $query->where('stripe_payment_method', 'like', '%' . $request->card . '%');
            })
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'] . ":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'] . ":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }


                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            });

        if ($user->hasRole('Super Admin')) {
            $customers = $customersQuery->latest('id')->paginate(20)->withQueryString();
        } elseif ($user->hasAnyRole($roles)) {
            if ($user->can('Customer-Global')) {

                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');

                $customers = $customersQuery
                    ->whereNotIn('created_by', $superAdminIds)
                    ->latest('id')->paginate(20)->withQueryString();

            } else {
                $customers = $customersQuery->whereHas('payments.link', function ($query) use ($user) {
                    $query->where('created_by', $user->id);
                })->latest('id')->paginate(20)->withQueryString();
            }

        } else {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $countries = CountryCurrencies::select('id', 'country', 'code', 'aplha_code2')->orderBy('country', 'ASC')->get();

        $exportColumns = $this->customerExport->columns();

        return Inertia::render('Admin/Customer/List', [
            'exportColumns' => $exportColumns,
            'columns' => $columns,
            'page' => $page,
            'customers' => $customers,
            'countries' => $countries,
        ]);
    }

    public function detail($clientID)
    {
        $customer = User::where('id', '=', $clientID)->with([
            'invoices' => function ($invoice) {
                $invoice->with([
                    'payment' => function ($payment) {
                        $payment->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'last_four', 'status', 'created_at');
                        $payment->with([
                            'link' => function ($link) {
                                $link->select('id', 'customer_id', 'token', 'item_name', 'price', 'currency', 'status');
                                $link->with('currencyCountry:id,aplha_code3,code,symbol');
                            }
                        ]);
                    }
                ]);
            },
            'countryData:id,aplha_code3,country,symbol',
            'quotes',
            'paymentLogs' => function ($logs) {
                $logs->latest('created_at');
            },
            'payments' => function ($query) {
                $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                $query->with([
                    'link' => function ($linkQ) {
                        $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                        $linkQ->with('currencyCountry:id,aplha_code3,code,symbol', 'gateway:id,name,gateway,statement_descriptor,environment');
                    },
                    'paymentLogs' => function ($logs) {
                        $logs->latest('created_at');
                    },
                ]);
                $query->latest('id');
                $query->paginate(5);
            }
        ])->firstOrFail();

        $succeededPayments = collect();
        $customer?->payments->map(function ($payment) use ($succeededPayments) {
            if ($payment->link->status == 2) {
                return $succeededPayments->push($payment->link);
            } else {
                return 0;
            }
        });

        $countries = CountryCurrencies::select('id', 'country', 'code', 'aplha_code2')->orderBy('country', 'ASC')->get();

        $relatedBrief = [];

        if($clientID){
            $relatedBrief = BriefHelper::getCustomerRelatedBriefs($clientID);
        }

        return Inertia::render('Admin/Customer/Detail', [
            'customer' => $customer,
            'total_spent' => $succeededPayments->sum('price'),
            'countries' => $countries,
            'relatedBrief' => $relatedBrief,
        ]);
    }

    public function update(Request $request, $customerID)
    {

        $rules = [
            'first_name' => 'nullable',
            'last_name' => 'nullable',
            'email' => 'nullable|email|unique:users,email,' . $customerID,
            'phone' => 'nullable|min:8|unique:users,phone,' . $customerID,
            'city' => 'nullable',
            'company' => 'nullable',
            'address' => 'nullable',
            'state' => 'nullable',
            'zipcode' => 'nullable',
            'country' => 'nullable',
            'image' => 'nullable',
            'stripe_customer_id' => 'nullable',
            'stripe' => 'nullable',
            'stripe_pm_id' => 'nullable',
            'stripe_payment_method' => 'nullable',
            'dashboard_data' => 'nullable',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()]);
        }

        if (!empty($request->phone)) {
            $requestPhone = str_replace(' ', '', $request->phone);
            $request->merge(['phone' => $requestPhone]);
        }

        $customer = User::where('id', '=', $customerID)->first();

        if ($request->metadata) {
            $metadata = Arr::pluck($request->metadata, 'value', 'key');

            $customerObj = json_decode($customer->stripe);

            if ($customerObj?->metadata) {
                $customerObj->metadata = $metadata;
            } else {
                $customerObj['metadata'] = $metadata;
            }

            $customer->update([
                'stripe' => json_encode($customerObj)
            ]);
        }

        $customer = User::findOrFail($customerID);

        if ($request->filled('first_name')) {
            $customer->update(['first_name' => $request->first_name]);
        }

        if ($request->filled('last_name')) {
            $customer->update(['last_name' => $request->last_name]);
        }

        if ($request->filled('email')) {
            $customer->update(['email' => $request->email]);
        }

        if ($request->filled('phone')) {
            $customer->update(['phone' => $request->phone]);
        }

        if ($request->filled('city')) {
            $customer->update(['city' => $request->city]);
        }

        if ($request->filled('company')) {
            $customer->update(['company' => $request->company]);
        }

        if ($request->filled('address')) {
            $customer->update(['address' => $request->address]);
        }

        if ($request->filled('state')) {
            $customer->update(['state' => $request->state]);
        }

        if ($request->filled('zipcode')) {
            $customer->update(['zipcode' => $request->zipcode]);
        }

        if ($request->filled('country')) {
            $customer->update(['country' => $request->country]);
        }

        if ($request->filled('image')) {
            $customer->update(['image' => $request->image]);
        }

        // $form = $request->only('first_name', 'last_name', 'email', 'phone', 'address', 'city', 'company', 'state', 'zipcode', 'country');
        // if (!empty($form)) {
        //     $customer->update($form);
        // }

        if ($request->returnJson) {

            // // Log the activity before logging out
            // ActivityLogHelper::staticLogActivity(
            //     'customer-info-updated',
            //     'customer info updated successfully'
            // );

            return response()->json(['success' => true, 'message' => 'Customer Updated Successful']);
        } else {
            return Redirect::route('admin.customer.detail', ['id' => $customer->id]);
        }
    }

    public function create(Request $request)
    {
        $data['success'] = true;

        if (!empty($request->phone)) {
            $requestPhone = str_replace(' ', '', $request->phone);
            $request->merge(['phone' => $requestPhone]);
        }

        try {
            $rules = [
                'first_name' => 'required',
                'last_name' => 'required',
                'email' => 'required|email|unique:users,email',
                'phone' => 'required|min:8|unique:users,phone',
                'city' => 'nullable',
                'company' => 'nullable',
                'address' => 'nullable',
                'state' => 'nullable',
                'zipcode' => 'nullable',
                'country' => 'nullable',
                'image' => 'nullable',
                'stripe_customer_id' => 'nullable',
                'stripe' => 'nullable',
                'stripe_pm_id' => 'nullable',
                'stripe_payment_method' => 'nullable',
                'dashboard_data' => 'nullable',
            ];

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()]);
            }

            $validatedData = $validator->validated();

            $validatedData['password'] = Hash::make('12345678');

            $newCustomer = $this->customer->create($validatedData);
            NotificationHelper::notify('create', 'New customer ' . ' ' . $newCustomer->email . ' ' . 'has been created', [
                'id' => $newCustomer->id,
                'data' => $newCustomer,
            ], Auth::id());

            // Log the activity before logging out
            // ActivityLogHelper::staticLogActivity(
            //     'customer-created',
            //     'new customer has been created successfully'
            // );

            $data['data'] = $newCustomer;
        } catch (Exception $e) {
            $data['success'] = $newCustomer;
            $data['message'] = $e->getMessage();
        }

        return response()->json($data);
    }

    public function find($customerString)
    {
        $data['success'] = true;
        try {
            $customers = User::select('id', 'first_name', 'email', 'city', 'company', 'address', 'state', 'zipcode', 'country')
                ->where('first_name', 'like', $customerString . '%')
                ->orWhere('last_name', 'like', $customerString . '%')
                ->orWhere('email', 'like', $customerString . '%')
                ->get();
            $data['data'] = $customers;
        } catch (Exception $e) {
            $data['success'] = false;
            $data['message'] = $e->getMessage();
        }

        return response()->json($data);
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:users,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:users,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        if ($request->user()->hasRole('Super Admin') || $request->user()->can('Customer-Global')) {
            try {
                if ($request->filled('id')) {
                    $customer = User::findOrFail($request->id);
                    $customer->delete();
                } else if ($request->filled('ids')) {
                    User::destroy($request->ids);
                }

                // Log the activity before logging out
                // ActivityLogHelper::staticLogActivity(
                //     'customer-deleted',
                //     'customer deleted successfully'
                // );

                return response()->json([
                    'success' => true,
                    'message' => "Customer(s) successfully deleted.",
                ], 200);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => "An error occurred while deleting customer(s).",
                ], 500);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => "You are not allowed to perform this operation.",
            ], 403);
        }
    }

    public function export(Request $request)
    {
        $data['success'] = false;
        $rules = [
            'columns' => 'nullable|array',
            'dateRange' => 'nullable|array',
            'rangeOption' => 'nullable',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $data['errors'] = $validator->errors();
            return response()->json($data, 422);
        }

        try {
            $validatedData = $validator->validated();
            $this->customerExport->columns();

            $data['success'] = true;
            $data['data'] = $this->customerExport->export($validatedData);
            return response()->json($data);
            //return Excel::download(new UsersExport($validatedData['columns'], $validatedData['dateRange'], $validatedData['rangeOption']), 'customers.xlsx');
        } catch (Exception $ex) {
            $data['message'] = $ex->getMessage();
            return response()->json($data, $ex->getCode());
        }
    }

    public function import(Request $request)
    {
        try {
            Excel::import(new CustomerImport, $request->file('file'));
            return response()->json(['success' => true, 'message' => 'File imported successfully']);
        } catch (\Exception $e) {
            // Handle other exceptions
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    // Block or allow customer for transaction
    public function allowOrBlockCustomer(Request $request, $id)
    {
        $response['success'] = true;

        $validator = Validator::make($request->all(), [
            'email' => 'nullable|email|unique:users,email',
            'isBlocked' => 'boolean'
        ]);

        if ($validator->fails()) {
            $response['errors'] = $validator->errors();

            return response()->json($response, 403);
        } else {
            try {
                $this->customer->update($id, [
                    'isBlocked' => ($request->isBlocked) ? 1 : 0
                ]);

                $response['success'] = true;

                if ($request->isBlocked) {
                    $response['message'] = 'Added to block list';
                } else {
                    $response['message'] = 'Remove from block list';
                }

                return response()->json($response);
            } catch (Exception $e) {
                $response['code'] = $e->getCode();
                $response['message'] = $e->getMessage();

                return response()->json($response, 500);
            }
        }
    }

    public function search(Request $request)
    {

        $customers = User::query()
            ->select('id', 'first_name', 'last_name', 'email', 'phone', 'city', 'company', 'address', 'state', 'zipcode', 'country', 'stripe_customer_id', 'stripe_payment_method', 'created_at', 'target')
            ->role('customer')
            ->latest('id')
            ->when(!empty($request->keyword), function ($query) use ($request) {
                $query->where(function ($q) use ($request) {
                    $q->where('first_name', 'like', '%' . $request->keyword . '%')
                      ->orWhere('last_name', 'like', '%' . $request->keyword . '%')
                      ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$request->keyword}%"])
                      ->orWhere('email', 'like', '%' . $request->keyword . '%');
                });
            })
            ->when(!empty($request->limit), function ($query) use ($request) {
                $query->limit($request->limit);
            })
            ->get();


        $customerData = [];

        foreach ($customers as $customer) {
            $customerData[$customer->id] = [
                'id' => $customer->id,
                'name' => $customer->first_name . ' ' . $customer->last_name,
                'email' => $customer->email,
                'phone' => $customer->phone,
                'companyName' => $customer->company
            ];
        }

        $sortedCustomerData = collect($customerData)->sortByDesc('id')->values()->all(); 

        return response()->json(['success' => true, 'message' => 'Customer retrived.', 'data' => $sortedCustomerData]);
    }

    public function updateMetadata(Request $request, $id) {

        $customer = User::where('id', '=', $id)->first();

        if($request->metadata) {
            $metadata = Arr::pluck($request->metadata, 'value', 'key');

            $customerObj = json_decode($customer->stripe);

            if ($customerObj?->metadata) {
                $customerObj->metadata = $metadata;
            } else {
                $customerObj['metadata'] = $metadata;
            }

            $customer->update([
                'stripe' => json_encode($customerObj)
            ]);
        }

        return Redirect::route('admin.customer.detail', [ 'id' => $customer->id ]);
    }
}
