<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\BriefHelper;
use App\Helper\FileHelper;
use App\Helper\LogsHelper;
use App\Helper\PDFHelper;
use App\Http\Controllers\Controller;
use App\Models\BriefActivity;
use App\Models\BriefLink;
use App\Models\User;
use App\Models\VisitedBrief;
use App\Models\WebBrief;
use App\Services\ApiService;
use App\Services\BinApiService;
use App\Services\BriefActivityService;
use App\Services\BriefLogService;
use App\Services\IpGeoLocationService;
use App\Services\LogService;
use App\Services\WebBriefService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class WebBriefController extends Controller
{
    private $webBrief, $webBriefService, $apiService, $briefLogService, $briefLink, $briefActivityService, $ipLocationService, $neutrinoApi, $logRepository;
    private $logsDirectory = 'webBrief', $linkLongDirectory = 'brief_link';

    public function __construct(WebBrief $webBrief, WebBriefService $webBriefService, ApiService $apiService, BriefLogService $briefLogService, BriefLink $briefLink, BriefActivityService $briefActivityService, IpGeoLocationService $ipLocationService, BinApiService $neutrinoApi, LogService $logRepository)
    {
        $this->webBrief = $webBrief;
        $this->webBriefService = $webBriefService;
        $this->apiService = $apiService;
        $this->briefLogService = $briefLogService;
        $this->briefLink = $briefLink;
        $this->briefActivityService = $briefActivityService;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->logRepository = $logRepository;
    }

    public function list(Request $request)
    {
        $briefs = $this->webBriefService->list($request);

        $page = [
            'title' => 'Web Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'web.brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'web.brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "business_name", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        return Inertia::render('Admin/Brief/Brief/WebList', [
            'briefs' => $briefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }

    public function detail(Request $request)
    {
        $visitedData = VisitedBrief::select('id', 'ip', 'brief_type', 'data', 'created_at')->where('brief_type', 'web_brief')->latest('id')->paginate(30)->withQueryString();

        // $header = [
        //     'heading' => 'Web Brief Questionnaire',
        // ];

        // $briefRecordsTab = [
        //     'route' => 'web.brief.list',
        //     'page' => 'Brief/Brief/WebList',
        // ];
        // $briefActivityTab = [
        //     'route' => 'web.brief.detail',
        //     'page' => 'Brief/Brief/WebDetail',
        // ];

        $page = [
            'title' => 'Web Brief Activities',
            'heading' => 'Web Briefs',
            'addRoute' => 'web.brief.form',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'web.brief-detail.detail',
                    'isActive' => false,
                ],
                [
                    'title' => 'Activity',
                    'route' => 'web.brief-detail.activity',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'web.brief-detail.details',
                //     'isActive' => false,
                // ],
            ],
        ];

        $briefFormLink = 'web-brief/form';

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'visitedData' => $visitedData,
            'page' => $page,
            'briefFormLink' => $briefFormLink
        ]);
    }

    public function store(Request $request)
    {
        $rules = [
            'email' => 'sometimes|required|email|max:255',
            'name' => 'sometimes|required|string|max:255',
            'phone' => 'sometimes|required|string|max:40',
            'business_name' => 'sometimes|required|string|max:400',
            'business_address' => 'sometimes|nullable|string|max:400',
            'business_phone' => 'sometimes|nullable|string|max:40',
            'business_email' => 'sometimes|nullable|email|max:400',
            'social_media_handles' => 'sometimes|nullable|string|max:400',
            'website' => 'sometimes|nullable|string|max:400',
            'web_hosting' => 'sometimes|nullable|string|max:400',
            'current_domain_address' => 'sometimes|nullable|string|max:400',
            'organization_operations' => 'sometimes|required|string|max:65535',
            'website_purpose' => 'sometimes|required|string|max:65535',
            'portray' => 'sometimes|required|array',
            'audience' => 'sometimes|nullable|string|max:65535',
            'pages' => 'sometimes|nullable|string|max:65535',
            'competitors' => 'sometimes|nullable|string|max:65535',
            'website1_url' => 'sometimes|nullable|string|max:400',
            'website1_comments' => 'sometimes|nullable|string|max:65535',
            'website2_url' => 'sometimes|nullable|string|max:400',
            'website2_comments' => 'sometimes|nullable|string|max:65535',
            'website3_url' => 'sometimes|nullable|string|max:400',
            'website3_comments' => 'sometimes|nullable|string|max:65535',
            'website_features' => 'sometimes|nullable|string|max:65535',
            'maintenance_team' => 'sometimes|nullable|string|max:65535',
            'website_content' => 'sometimes|required|string|max:65535',
            'slogan' => 'sometimes|nullable|string|max:400',
            'have_logo' => 'sometimes|required|string|max:65535',
            'redesign_logo' => 'sometimes|required|string|max:65535',
            'website_fonts' => 'sometimes|nullable|string|max:400',
            'website_colors' => 'sometimes|nullable|string|max:400',
            'anything_else' => 'sometimes|nullable|string|max:65535',
        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Full Name is required.',
            'phone.required' => 'Phone number is required.',
            'business_name.required' => 'Business name is required.',
            'business_email.email' => 'Email must be a valid email address.',
            'website_purpose.required' => 'Purpose of your website is required.',
            'portray.required' => 'Website Portray is required.',
            'organization_operations.required' => 'Brief Describe Organization Operations is required.',
            'website_content.required' => 'Website Content availability is required.',
            'have_logo.required' => 'Logo availability is required.',
            'redesign_logo.required' => 'Redesigning your logo is required.',

        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'company' => $request->business_name,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }

            $webBrief = $this->webBriefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $webBrief['encoded_id']);

            $logData = [
                'activity' => 'Web Brief has been created by ' . $webBrief['email'] . '.',
                'request' => $request,
                'response' => response()->json($webBrief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $webBrief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $webBrief['id'],
                'class_type' => get_class(new WebBrief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'Web Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($webBrief)->content(),
                'response' => 201,
                'code' => response()->json($webBrief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->briefActivityService->create($request, $webBrief['id'], 'web_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if($webBrief['encoded_id']){
                $foundBrief = $this->briefLink->where('id', $webBrief['brief_link_id'])->first();
                $foundBrief->customer_id = $request->customer_id;
                $foundBrief->status = "submitted";
                $foundBrief->save();
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if($webBrief['id']){
                $this->webBrief->find($webBrief['id'])
                ->update([
                    'location' => $locationData,
                    'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                ]);
            }

            return redirect()->route('web.brief.success')->with('success', 'Web Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'Web Design Questionnaire',
            'description' => ['Your response has been recorded.']
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function stringToArray($string)
    {
        $array = explode(",", $string);
        return $array;
    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->webBriefService->fetchByEncodedID($encoded_id, true);

            $formData['portray'] = $this->stringToArray($formData['portray']);

            $htmlContent = Blade::render('brief.web_download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf'; // Adjust the file name as needed

            // $this->briefLogService->log([
            //     'brief_type' => 'web_brief',
            //     'brief_id' => $formData['id'],
            //     'activity' => 'Web Brief downloaded.',
            //     'type' => 'web_brief.download',
            // ]);

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new WebBrief()),
                'activity' => 'Web Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'web_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = WebBrief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return  $e->getMessage();
        }
    }

    public function view(Request $request)
    {
        $isDisabled = true;
        if (Auth::check()) {
            $brief = $this->webBriefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->webBriefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }


        return Inertia::render('WebBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

    public function briefDetail(Request $request)
    {

        $brief = $this->webBrief->with('briefLogs:id,id,brief_id,activity,type,created_by,request,response,pin,code,event,created_at', 'briefLink:id,link,brief_type,salesperson_id,status,visited,created_at', 'customer:id,first_name,last_name,email,phone,address')->findOrFail($request->id);

        $relatedBrief = [];
        $succeededPayments = collect();


        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'web');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                'payments' => function ($query) {
                    $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                    $query->with([
                        'link' => function ($linkQ) {
                            $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                            $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                        }
                    ]);
                    $query->latest('id')->get();
                }
            ])->firstOrFail();

            $customer?->payments->each(function ($payment) use ($succeededPayments) {
                if ($payment->link->status == 2) { // Assuming '2' is the status for successful payments
                    $succeededPayments->push($payment->link);
                }
            });
        }

        $briefDetails = [
            'project_details' => [
                'business_name' => $brief->business_name,
                'address' => $brief->business_address,
                'contact' => $brief->phone,
                'email' => $brief->email,
            ],
            'project_requirements' => [
                'social_media_URL' => $brief->social_media_handles,
                'current_website' => $brief->website,
                'current_hosting' => $brief->web_hosting,
                'current_domain' => $brief->current_domain_address,
                'organization_overview' => $brief->organization_operations,
                'purpose' => $brief->website_purpose,
                'overall_feeling' => $brief->portray,
                'target_audience' => $brief->audience,
                'pages' => $brief->pages,
                'competitors' => $brief->competitors,
                'website_1' => $brief->website1_url,
                'website_1_comments' => $brief->website1_comments,
                'website_2' => $brief->website2_url,
                'website_2_comments' => $brief->website2_comments,
                'website_3' => $brief->website3_url,
                'website_3_comments' => $brief->website3_comments,
                'features_/_plugins' => $brief->website_features,
                'need_maintenance_team' => $brief->maintenance_team,
                'content' => $brief->website_content,
                'tagline_/_slogan' =>  $brief->slogan,
                'have_a_high_resolution_logo?' =>  $brief->have_logo,
                'redesigning_logo' =>  $brief->redesign_logo,
                'fonts' => $brief->website_fonts,
                'colors' => $brief->website_colors,
                'comments' => $brief->anything_else,
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);

        $formSubmissionAccuracy = 100;

        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy -= $errorLogsCount;
        $formSubmissionAccuracy = max($formSubmissionAccuracy, 0);

        $briefActivities = VisitedBrief::where('brief_type', 'web_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'Web Brief Detail',
            'heading' => 'Web Brief Details',
            'route' => 'web.brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'web.brief.updateMetadata',
            'deleteRoute' => 'web.brief.delete',
            'formRoute' => 'frontend.web.brief.view',
            'activityRoute' => 'web.brief-detail.activity',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'web.brief-detail.detail',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'web.brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'web.brief-detail.details',
                //     'isActive' => false,
                // ]
            ],
        ];

        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'web_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'web_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID' => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }

    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'Web Brief Activity',
            'heading' => 'Web Brief Activities',
            'route' => 'web.brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'Web Brief Activity',
        //     'heading' => 'Web Brief Activities',
        //     'route' => 'web.brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'web.brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'web.brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'web.brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'web_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }

    // public function details(Request $request){
    //     $brief = $this->webBriefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'Web Brief Detail',
    //         'heading' => 'Web Brief Details',
    //         'route' => 'web.brief-detail.detail',
    //         'formRoute' => 'frontend.web.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'web.brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'web.brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'web.brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'email' => $brief->briefLink->customer->email,
    //                 'full_name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //                 'mobile_number' => $brief->briefLink->customer->phone,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'business_name' => $brief->business_name,
    //                 'business_address' => $brief->business_address,
    //                 'business_contact_number' => $brief->business_phone,
    //                 'business_email' => $brief->business_email,
    //                 'social_media_handles' => $brief->social_media_handles,
    //                 'current_website_URL_(if applicable)' => $brief->website,
    //                 'current_website_hosting_company_(if applicable)' => $brief->web_hosting,
    //                 'current_domain_address_(if applicable)' => $brief->current_domain_address,
    //                 'please_describe_your_organization\'s_operations_in_a_few_sentences' => $brief->organization_operations,
    //                 'purpose_of_your_website' => explode(',', $brief->website_purpose),
    //                 'overall_feeling_that_you_would_like_to_portray_with_your_new_website' => explode(',', $brief->portray),
    //                 'target_audience' => $brief->audience,
    //                 'pages_you_require_(other_then_the_homepage),_eg,_about_us,_services/products,_contact_us' => $brief->pages,
    //                 'similar_businesses_/_competitors' => $brief->competitors,
    //                 'competitor_website_#1_URL' => $brief->website1_url,
    //                 'comments_on_website_#1' => $brief->website1_comments,
    //                 'competitor_website_#2_URL' => $brief->website2_url,
    //                 'comments_on_website_#2' => $brief->website2_comments,
    //                 'competitor_website_#3_URL' => $brief->website3_url,
    //                 'comments_on_website_#3' => $brief->website3_comments,
    //                 'any_specific_features_/_plugins_you_plain_to_incorporate_into_your_website?' => $brief->website_features,
    //                 'do_you_plain_to_hire_a_maintenance_team_to_update_your_website?' => $brief->maintenance_team,
    //                 'do_you_currently_have_content_that_you_want_to_use_on_your_new_website?' => explode(',', $brief->website_content),
    //                 'do_you_have_a_tagline_/_slogan?' => $brief->slogan,
    //                 'do_you_have_a_high-resolution_logo?' => explode(',', $brief->have_logo),
    //                 'have_you_considered_redesigning_your_logo?' => explode(',', $brief->redesign_logo),
    //                 'what_fonts_do_you_typically_user?' => $brief->website_fonts,
    //                 'are_there_any_colors_that_you_want_to_incoporated_into_the_website?' => $brief->website_colors,
    //                 'anything_else' => $brief->anything_else,
    //             ],
    //         ]
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:web_briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:web_briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $webbrief = WebBrief::findOrFail($request->id);
                $webbrief->delete();
            } else if ($request->filled('ids')) {
                WebBrief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'web_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'web_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }
    }

    public function updateMetadata(Request $request){
        try {
            $query = $this->webBrief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = WebBrief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'web_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }
}
