<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use PDO;

class BackupDatabase extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'backup:database';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a backup of the database';
    
    const CHUNK_SIZE = 1000;
    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        ini_set('memory_limit', '1G');
        ini_set('max_execution_time', '0'); // No time limit

        $databaseName = env('DB_DATABASE');
        $backupFile = storage_path('app/backups/db/' . $databaseName . '.sql');

        if (!file_exists(dirname($backupFile))) {
            mkdir(dirname($backupFile), 0755, true);
        }

        $fileHandle = fopen($backupFile, 'w');
        fwrite($fileHandle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        $pdo = DB::connection()->getPdo();
        $tables = $this->getAllTables($pdo);

        foreach ($tables as $table) {
            $this->backupTableStructure($pdo, $fileHandle, $table);
            $this->backupTableDataInChunks($pdo, $fileHandle, $table);
        }

        fwrite($fileHandle, "\nSET FOREIGN_KEY_CHECKS = 1;\n\n");
        fclose($fileHandle);

        $this->info("Database backup created successfully at: $backupFile");

        $this->deleteOldBackups();

        return 0;
    }

    private function getAllTables(PDO $pdo)
    {
        $tablesQuery = $pdo->prepare("SHOW TABLES");
        $tablesQuery->execute();
        return $tablesQuery->fetchAll(PDO::FETCH_COLUMN);
    }

    private function backupTableStructure(PDO $pdo, $fileHandle, $table)
    {
        $showTableQuery = $pdo->prepare("SHOW CREATE TABLE `$table`");
        $showTableQuery->execute();
        $createTableResult = $showTableQuery->fetch(PDO::FETCH_ASSOC);

        fwrite($fileHandle, "\n\n" . $createTableResult['Create Table'] . ";\n\n");
    }

    private function backupTableDataInChunks(PDO $pdo, $fileHandle, $table)
    {
        $offset = 0;
    
        while (true) {
            $selectQuery = $pdo->prepare("SELECT * FROM `$table` LIMIT " . self::CHUNK_SIZE . " OFFSET " . $offset);
            $selectQuery->execute();
            $rows = $selectQuery->fetchAll(PDO::FETCH_ASSOC);
    
            if (empty($rows)) {
                break; // Exit if no more rows
            }
    
            $insertStatements = []; // Collect insert statements for bulk writing
    
            foreach ($rows as $row) {
                $columns = array_keys($row);
                $values = array_map(function ($value) {
                    return is_null($value) ? 'NULL' : "'" . addslashes($value) . "'"; // Handle NULL and escape values
                }, array_values($row));
    
                $insertStatement = "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $values) . ");";
                $insertStatements[] = $insertStatement; // Add to the array
            }
    
            // Write all insert statements at once
            fwrite($fileHandle, implode("\n", $insertStatements) . "\n");
    
            $offset += self::CHUNK_SIZE; // Move to the next set of rows
        }
    }

    private function quoteValue($value)
    {
        return is_null($value) ? 'NULL' : "'" . str_replace("'", "''", $value) . "'";
    }


      private function deleteOldBackups()
    {
        $backupDir = storage_path("app/backups/db/");
        $files = glob("{$backupDir}*.sql");

        if (count($files) > 1) {
            array_multisort(array_map('filemtime', $files), SORT_NUMERIC, SORT_DESC, $files);
            array_shift($files);
            foreach ($files as $file) {
                unlink($file);
                $this->info("Deleted old backup: {$file}");
            }
        }
    }
}
